"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.S8VProvider = void 0;
const axios_1 = __importDefault(require("axios"));
const verification_types_1 = require("./verification.types");
class S8VProvider {
    client;
    config;
    constructor(config) {
        this.config = config;
        const providerTimeout = Math.min(config.timeout || 25000, 25000);
        this.client = axios_1.default.create({
            baseURL: config.baseUrl || 'https://www.s8v.ng/api',
            timeout: providerTimeout,
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${config.apiKey}`,
            },
        });
        this.client.interceptors.request.use((config) => {
            console.log(`[S8VProvider] ${config.method?.toUpperCase()} ${config.url}`);
            return config;
        }, (error) => {
            console.error('[S8VProvider] Request error:', error);
            return Promise.reject(error);
        });
        this.client.interceptors.response.use((response) => response, (error) => {
            console.error('[S8VProvider] Response error:', error.response?.data || error.message);
            return Promise.reject(error);
        });
    }
    getProviderInfo() {
        return {
            name: 'S8V',
            version: '1.0.0',
            baseUrl: this.config.baseUrl || 'https://www.s8v.ng/api',
            features: ['nin_verification'],
        };
    }
    updateConfig(config) {
        this.config = { ...this.config, ...config };
        if (config.baseUrl) {
            this.client.defaults.baseURL = config.baseUrl;
        }
        if (config.apiKey) {
            this.client.defaults.headers['Authorization'] = `Bearer ${config.apiKey}`;
        }
        if (config.timeout) {
            const providerTimeout = Math.min(config.timeout, 25000);
            this.client.defaults.timeout = providerTimeout;
        }
    }
    async verifyNin(request) {
        try {
            this.validateNinRequest(request);
            const payload = {
                token: this.config.apiKey,
                nin: request.nin
            };
            console.log(`[S8VProvider] Verifying NIN: ${request.nin}`);
            const response = await this.client.post('/verification', payload);
            if (response.data.error) {
                const errorMessage = response.data.error || 'Unknown error';
                if (errorMessage.toLowerCase().includes('invalid') || errorMessage.toLowerCase().includes('not found')) {
                    throw new verification_types_1.ProviderError('NIN not found or invalid', 404);
                }
                if (errorMessage.toLowerCase().includes('unauthorized') || errorMessage.toLowerCase().includes('invalid token')) {
                    throw new verification_types_1.AuthenticationError('Invalid API token');
                }
                throw new verification_types_1.ProviderError(errorMessage, 400);
            }
            const transformedResponse = {
                batchid: response.data.batchid || '',
                birthcountry: response.data.birthcountry || 'Nigeria',
                birthdate: response.data.birthdate || response.data.date_of_birth || response.data.dateOfBirth || '',
                birthlga: response.data.birthlga || response.data.lga || response.data.lga_name || '',
                birthstate: response.data.birthstate || response.data.state || '',
                cardstatus: response.data.cardstatus || 'Active',
                centralID: response.data.centralID || request.nin,
                educationallevel: response.data.educationallevel || '',
                email: response.data.email || '',
                emplymentstatus: response.data.emplymentstatus || '',
                firstname: response.data.firstname || response.data.first_name || response.data.firstName || '',
                gender: response.data.gender || '',
                heigth: response.data.heigth || '',
                maritalstatus: response.data.maritalstatus || '',
                middlename: response.data.middlename || response.data.middle_name || response.data.middleName || '',
                nin: request.nin || '',
                nok_address1: response.data.nok_address1 || response.data.address || '',
                nok_firstname: response.data.nok_firstname || '',
                nok_lga: response.data.nok_lga || '',
                nok_middlename: response.data.nok_middlename || '',
                nok_postalcode: response.data.nok_postalcode || '',
                nok_state: response.data.nok_state || '',
                nok_surname: response.data.nok_surname || '',
                nok_town: response.data.nok_town || '',
                nspokenlang: response.data.nspokenlang || '',
                ospokenlang: response.data.ospokenlang || '',
                pfirstname: response.data.pfirstname || '',
                photo: response.data.photo || response.data.photo_url || '',
                pmiddlename: response.data.pmiddlename || '',
                profession: response.data.profession || response.data.poccupation || '',
                psurname: response.data.psurname || response.data.last_name || response.data.lastName || '',
                religion: response.data.religion || '',
                residence_AdressLine1: response.data.residence_AdressLine1 || response.data.residence_Adressline1 || response.data.address || '',
                residence_Town: response.data.residence_Town || '',
                residence_lga: response.data.residence_lga || response.data.lga || response.data.lga_name || '',
                residence_state: response.data.residence_state || response.data.state || '',
                residencestatus: response.data.residencestatus || response.data.residence_status || '',
                signature: response.data.signature || '',
                surname: response.data.surname || response.data.last_name || response.data.lastName || '',
                telephoneno: response.data.telephoneno || response.data.telephone || response.data.phone_number || response.data.phoneNumber || '',
                title: response.data.title || '',
                trackingId: response.data.tracking_id || response.data.trackingId || this.generateTrackingId(),
            };
            return transformedResponse;
        }
        catch (error) {
            if (error instanceof verification_types_1.ValidationError || error instanceof verification_types_1.AuthenticationError || error instanceof verification_types_1.ProviderError) {
                throw error;
            }
            if (error && typeof error === 'object' && 'response' in error) {
                const axiosError = error;
                const status = axiosError.response?.status;
                const data = axiosError.response?.data;
                if (status === 401) {
                    throw new verification_types_1.AuthenticationError('Invalid API credentials');
                }
                if (status === 404) {
                    throw new verification_types_1.ProviderError('NIN not found', 404);
                }
                if (status === 429) {
                    throw new verification_types_1.ProviderError('Rate limit exceeded, please try again later', 429);
                }
                if (status >= 500) {
                    throw new verification_types_1.ProviderError('S8V service temporarily unavailable', 503);
                }
                throw new verification_types_1.ProviderError(data?.error || data?.message || `HTTP ${status} error`);
            }
            throw new verification_types_1.ProviderError(`NIN verification failed: ${error instanceof Error ? error.message : String(error)}`);
        }
    }
    async verifyBvn(request) {
        throw new verification_types_1.ProviderError('BVN verification not supported by S8V provider');
    }
    async createIpeClearance(request) {
        throw new verification_types_1.ProviderError('IPE clearance not supported by S8V provider');
    }
    async getIpeClearanceStatus(transactionId) {
        throw new verification_types_1.ProviderError('IPE clearance status check not supported by S8V provider');
    }
    async createPersonalization(request) {
        throw new verification_types_1.ProviderError('Document personalization not supported by S8V provider');
    }
    async healthCheck() {
        try {
            const testPayload = {
                token: this.config.apiKey,
                nin: '00000000000'
            };
            const response = await this.client.post('/verification', testPayload, {
                timeout: 5000
            });
            return response.status >= 200 && response.status < 500;
        }
        catch (error) {
            console.error('[S8VProvider] Health check failed:', error);
            return false;
        }
    }
    validateNinRequest(request) {
        if (!request.nin) {
            throw new verification_types_1.ValidationError('NIN is required for S8V verification');
        }
        if (!/^\d{11}$/.test(request.nin)) {
            throw new verification_types_1.ValidationError('NIN must be 11 digits');
        }
    }
    generateTrackingId() {
        return `s8v_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    }
}
exports.S8VProvider = S8VProvider;
