"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RetryStrategy = void 0;
class RetryStrategy {
    config;
    constructor(config = {}) {
        this.config = {
            maxRetries: 3,
            baseDelay: 1000,
            maxDelay: 10000,
            backoffMultiplier: 2,
            jitter: true,
            ...config
        };
    }
    async execute(operation, shouldRetry) {
        let lastError;
        for (let attempt = 0; attempt <= this.config.maxRetries; attempt++) {
            try {
                return await operation();
            }
            catch (error) {
                lastError = error;
                if (attempt === this.config.maxRetries || !this.shouldRetryError(error, shouldRetry)) {
                    throw error;
                }
                const delay = this.calculateDelay(attempt);
                console.log(`[RetryStrategy] Attempt ${attempt + 1} failed, retrying in ${delay}ms`);
                await this.sleep(delay);
            }
        }
        throw lastError;
    }
    shouldRetryError(error, customShouldRetry) {
        if (customShouldRetry) {
            return customShouldRetry(error);
        }
        if (error?.response?.status) {
            const status = error.response.status;
            return status >= 500 || status === 408 || status === 429;
        }
        if (error?.code === 'ECONNABORTED' || error?.code === 'ENOTFOUND' || error?.code === 'ECONNRESET') {
            return true;
        }
        return false;
    }
    calculateDelay(attempt) {
        let delay = this.config.baseDelay * Math.pow(this.config.backoffMultiplier, attempt);
        delay = Math.min(delay, this.config.maxDelay);
        if (this.config.jitter) {
            const jitterRange = delay * 0.1;
            delay += (Math.random() - 0.5) * 2 * jitterRange;
        }
        return Math.max(0, Math.floor(delay));
    }
    sleep(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }
}
exports.RetryStrategy = RetryStrategy;
