"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.verificationProviderManager = exports.VerificationProviderManager = void 0;
const env_1 = require("../../config/env");
const unify_provider_1 = require("./unify.provider");
const mock_provider_1 = require("./mock.provider");
const s8v_provider_1 = require("./s8v.provider");
const verification_types_1 = require("./verification.types");
class VerificationProviderManager {
    static instance;
    providers = {};
    currentProvider = 'unify';
    sandboxConfig;
    constructor() {
        this.sandboxConfig = {
            enabled: env_1.env.VERIFICATION_SANDBOX_MODE || false,
            mockDelay: 1000,
            simulateErrors: false,
            errorRate: 0.1,
        };
        this.initializeProviders();
    }
    static getInstance() {
        if (!VerificationProviderManager.instance) {
            VerificationProviderManager.instance = new VerificationProviderManager();
        }
        return VerificationProviderManager.instance;
    }
    setProvider(providerName) {
        if (!this.providers[providerName]) {
            throw new verification_types_1.ValidationError(`Provider '${providerName}' is not registered`);
        }
        this.currentProvider = providerName;
        console.log(`[VerificationProviderManager] Switched to provider: ${providerName}`);
    }
    getCurrentProvider() {
        return this.currentProvider;
    }
    getProviderInfo() {
        const provider = this.providers[this.currentProvider];
        return provider ? provider.getProviderInfo() : null;
    }
    getCurrentProviderName() {
        return this.currentProvider;
    }
    enableSandboxMode() {
        this.sandboxConfig.enabled = true;
        this.currentProvider = 'mock';
        console.log('[VerificationProviderManager] Sandbox mode enabled');
    }
    disableSandboxMode() {
        this.sandboxConfig.enabled = false;
        this.currentProvider = 'unify';
        console.log('[VerificationProviderManager] Sandbox mode disabled');
    }
    isSandboxMode() {
        return this.sandboxConfig.enabled;
    }
    updateSandboxConfig(config) {
        this.sandboxConfig = { ...this.sandboxConfig, ...config };
        console.log('[VerificationProviderManager] Sandbox config updated:', this.sandboxConfig);
    }
    getSandboxConfig() {
        return { ...this.sandboxConfig };
    }
    async verifyNin(request) {
        try {
            const provider = this.getCurrentProviderInstance();
            return await provider.verifyNin(request);
        }
        catch (error) {
            console.error('[VerificationProviderManager] NIN verification failed:', error);
            throw error;
        }
    }
    async verifyBvn(request) {
        try {
            const provider = this.getCurrentProviderInstance();
            return await provider.verifyBvn(request);
        }
        catch (error) {
            console.error('[VerificationProviderManager] BVN verification failed:', error);
            throw error;
        }
    }
    async createIpeClearance(request) {
        try {
            const provider = this.getCurrentProviderInstance();
            return await provider.createIpeClearance(request);
        }
        catch (error) {
            console.error('[VerificationProviderManager] IPE clearance creation failed:', error);
            throw error;
        }
    }
    async getIpeClearanceStatus(transactionId) {
        try {
            const provider = this.getCurrentProviderInstance();
            return await provider.getIpeClearanceStatus(transactionId);
        }
        catch (error) {
            console.error('[VerificationProviderManager] IPE clearance status check failed:', error);
            throw error;
        }
    }
    async createPersonalization(request) {
        try {
            const provider = this.getCurrentProviderInstance();
            return await provider.createPersonalization(request);
        }
        catch (error) {
            console.error('[VerificationProviderManager] Personalization creation failed:', error);
            throw error;
        }
    }
    async healthCheck() {
        try {
            const provider = this.getCurrentProviderInstance();
            return await provider.healthCheck();
        }
        catch (error) {
            console.error('[VerificationProviderManager] Health check failed:', error);
            return false;
        }
    }
    getRegisteredProviders() {
        return Object.keys(this.providers);
    }
    setCurrentProvider(provider) {
        if (!this.providers[provider]) {
            throw new verification_types_1.ValidationError(`Provider '${provider}' is not available`);
        }
        this.currentProvider = provider;
        console.log(`[VerificationProviderManager] Switched to provider: ${provider}`);
    }
    async getProviderStatus() {
        const status = {};
        for (const [name, provider] of Object.entries(this.providers)) {
            try {
                const isHealthy = await provider.healthCheck();
                const info = provider.getProviderInfo();
                status[name] = {
                    healthy: isHealthy,
                    info,
                    isCurrent: name === this.currentProvider,
                };
            }
            catch (error) {
                status[name] = {
                    healthy: false,
                    error: error instanceof Error ? error.message : String(error),
                    isCurrent: name === this.currentProvider,
                };
            }
        }
        return status;
    }
    initializeProviders() {
        const unifyConfig = {
            baseUrl: env_1.env.UNIFY_BASE_URL || 'https://unifyverifier.com',
            apiKey: '',
            timeout: 30000,
            retries: 3,
            sandboxMode: false,
        };
        this.providers['unify'] = new unify_provider_1.UnifyProvider(unifyConfig);
        const mockConfig = {
            baseUrl: 'mock://localhost',
            apiKey: 'mock-api-key',
            timeout: 5000,
            retries: 1,
            sandboxMode: true,
        };
        this.providers['mock'] = new mock_provider_1.MockProvider(mockConfig);
        const s8vConfig = {
            baseUrl: env_1.env.S8V_BASE_URL || 'https://www.s8v.ng/api',
            apiKey: env_1.env.S8V_API_KEY || '',
            timeout: 30000,
            retries: 3,
            sandboxMode: false,
        };
        this.providers['s8v'] = new s8v_provider_1.S8VProvider(s8vConfig);
        this.currentProvider = this.sandboxConfig.enabled ? 'mock' : 'unify';
        console.log(`[VerificationProviderManager] Initialized with provider: ${this.currentProvider}`);
    }
    getCurrentProviderInstance() {
        const provider = this.providers[this.currentProvider];
        if (!provider) {
            throw new verification_types_1.ProviderError(`Provider '${this.currentProvider}' is not available`);
        }
        return provider;
    }
    updateCurrentProviderConfig(config) {
        const providerInstance = this.getCurrentProviderInstance();
        if (typeof providerInstance.updateConfig === 'function') {
            providerInstance.updateConfig(config);
            console.log('[VerificationProviderManager] Updated provider config for', this.currentProvider);
        }
        else {
            console.warn('[VerificationProviderManager] Provider does not support runtime config update:', this.currentProvider);
        }
    }
    updateUnifyProviderConfig(apiKey, baseUrl) {
        console.log('[ProviderManager] Updating Unify provider config:', {
            apiKey: apiKey ? `${apiKey.substring(0, 8)}...` : 'MISSING',
            apiKeyLength: apiKey ? apiKey.length : 0,
            baseUrl: baseUrl || 'NOT_PROVIDED'
        });
        const config = {
            apiKey: apiKey,
            ...(baseUrl && { baseUrl })
        };
        if (this.providers['unify'] && typeof this.providers['unify'].updateConfig === 'function') {
            this.providers['unify'].updateConfig(config);
            console.log('[VerificationProviderManager] Updated Unify provider with new API key');
        }
        else {
            console.warn('[VerificationProviderManager] Unify provider does not support config updates');
        }
    }
    createProviderInstance(providerName, overrides) {
        const base = this.providers[providerName];
        if (!base) {
            throw new verification_types_1.ValidationError(`Provider '${providerName}' is not registered`);
        }
        const info = base.getProviderInfo();
        const defaultConfig = {
            baseUrl: info?.baseUrl || (providerName === 'mock' ? 'mock://localhost' : ''),
            apiKey: '',
            timeout: 25000,
            retries: 3,
            sandboxMode: providerName === 'mock',
        };
        const merged = { ...defaultConfig, ...(overrides || {}) };
        if (providerName === 'unify') {
            const { UnifyProvider } = require('./unify.provider');
            return new UnifyProvider(merged);
        }
        if (providerName === 'mock') {
            const { MockProvider } = require('./mock.provider');
            return new MockProvider(merged);
        }
        if (providerName === 's8v') {
            const { S8VProvider } = require('./s8v.provider');
            return new S8VProvider(merged);
        }
        throw new verification_types_1.ValidationError(`Provider '${providerName}' is not supported for dynamic creation`);
    }
}
exports.VerificationProviderManager = VerificationProviderManager;
exports.verificationProviderManager = VerificationProviderManager.getInstance();
