"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MockProvider = void 0;
const verification_types_1 = require("./verification.types");
class MockProvider {
    config;
    mockData = new Map();
    transactionCounter = 1;
    constructor(config) {
        this.config = config;
        this.initializeMockData();
    }
    getProviderInfo() {
        return {
            name: 'Mock Verification Provider',
            version: '1.0.0',
            features: ['nin', 'bvn', 'ipe', 'personalization'],
            baseUrl: 'mock://localhost',
        };
    }
    updateConfig(config) {
        this.config = { ...this.config, ...config };
    }
    async verifyNin(request) {
        await this.simulateDelay();
        this.validateNinRequest(request);
        if (this.shouldSimulateError()) {
            throw new verification_types_1.ProviderError('Mock NIN verification failed - simulated error');
        }
        const mockResponse = {
            batchid: this.generateRandomString(8),
            birthcountry: 'Nigeria',
            birthdate: request.dob || '15-06-1990',
            birthlga: 'Sokoto South',
            birthstate: 'Sokoto',
            cardstatus: 'Valid',
            centralID: this.generateRandomString(10),
            educationallevel: 'Tertiary',
            email: 'mock@example.com',
            emplymentstatus: 'Employed',
            firstname: request.first_name?.toUpperCase() || 'JOHN',
            gender: request.gender || 'm',
            heigth: '175cm',
            maritalstatus: 'Single',
            middlename: 'MOCK',
            nin: request.nin || this.generateRandomNIN(),
            nok_address1: '123 Mock Street',
            nok_firstname: 'MOCK',
            nok_lga: 'Sokoto South',
            nok_middlename: 'NOK',
            nok_postalcode: '840001',
            nok_state: 'Sokoto',
            nok_surname: 'CONTACT',
            nok_town: 'Sokoto',
            nspokenlang: 'English',
            ospokenlang: 'Hausa',
            pfirstname: 'JOHN',
            photo: 'data:image/jpg;base64,mock_photo_data',
            pmiddlename: 'MOCK',
            profession: 'Software Engineer',
            psurname: 'DOE',
            religion: 'Islam',
            residence_AdressLine1: '456 Mock Residence',
            residence_Town: 'Sokoto',
            residence_lga: 'Sokoto South',
            residence_state: 'Sokoto',
            residencestatus: 'Owned',
            signature: 'data:image/jpg;base64,mock_signature_data',
            surname: request.last_name?.toUpperCase() || 'DOE',
            telephoneno: request.phone || '08012345678',
            title: 'Mr',
            trackingId: this.generateTrackingId(),
        };
        return mockResponse;
    }
    async verifyBvn(request) {
        await this.simulateDelay();
        if (!request.BVN || request.BVN.length !== 11) {
            throw new verification_types_1.ValidationError('BVN must be exactly 11 digits');
        }
        if (this.shouldSimulateError()) {
            throw new verification_types_1.ProviderError('Mock BVN verification failed - simulated error');
        }
        const mockResponse = {
            adverseMediaReport: null,
            allValidationPassed: true,
            amlReport: null,
            businessId: this.generateRandomString(20),
            country: 'NG',
            createdAt: new Date().toISOString(),
            dataValidation: true,
            dateOfBirth: '1990-06-15',
            enrollmentBranch: 'Mock Branch',
            enrollmentInstitution: 'Mock Bank',
            firstName: 'John',
            gender: 'Male',
            id: this.generateRandomString(20),
            idNumber: this.generateRandomString(15),
            image: 'data:image/jpg;base64,mock_bvn_image',
            isConsent: true,
            lastModifiedAt: new Date().toISOString(),
            lastName: 'Doe',
            metadata: {},
            middleName: 'Mock',
            mobile: '08012345678',
            nin: this.generateRandomNIN(),
            parentId: null,
            reason: null,
            requestedAt: new Date().toISOString(),
            requestedById: this.generateRandomString(20),
            selfieValidation: true,
            shouldRetrivedNin: true,
            status: 'found',
            type: 'bvn',
        };
        return mockResponse;
    }
    async createIpeClearance(request) {
        await this.simulateDelay();
        if (!request.type || !request.tracking_ids || !Array.isArray(request.tracking_ids) || request.tracking_ids.length === 0) {
            throw new verification_types_1.ValidationError('Invalid IPE clearance request. Type and tracking_ids array are required.');
        }
        if (this.shouldSimulateError()) {
            throw new verification_types_1.ProviderError('Mock IPE clearance creation failed - simulated error');
        }
        const transactionId = this.transactionCounter++;
        this.mockData.set(`ipe_${transactionId}`, {
            type: request.type,
            tracking_ids: request.tracking_ids,
            status: 'pending',
            createdAt: new Date(),
        });
        return {
            success: 'Transaction created successfully!',
            trans_id: transactionId,
        };
    }
    async getIpeClearanceStatus(transactionId) {
        await this.simulateDelay();
        if (!transactionId || transactionId <= 0) {
            throw new verification_types_1.ValidationError('Invalid transaction ID');
        }
        const transactionKey = `ipe_${transactionId}`;
        const transaction = this.mockData.get(transactionKey);
        if (!transaction) {
            throw new verification_types_1.ProviderError('Transaction not found');
        }
        const timeSinceCreation = Date.now() - transaction.createdAt.getTime();
        const twoMinutes = 2 * 60 * 1000;
        if (timeSinceCreation > twoMinutes) {
            transaction.status = 'completed';
            this.mockData.set(transactionKey, transaction);
            const results = {};
            transaction.tracking_ids.forEach((trackingId) => {
                results[trackingId] = this.generateRandomNIN();
            });
            return results;
        }
        else {
            const results = {};
            transaction.tracking_ids.forEach((trackingId) => {
                results[trackingId] = '';
            });
            return results;
        }
    }
    async createPersonalization(request) {
        await this.simulateDelay();
        if (!request.type || !request.tracking_id) {
            throw new verification_types_1.ValidationError('Invalid personalization request. Type and tracking_id are required.');
        }
        if (this.shouldSimulateError()) {
            throw new verification_types_1.ProviderError('Mock personalization creation failed - simulated error');
        }
        return {
            address: '123 Mock Street, Mock City',
            face: 'data:image/jpg;base64,mock_face_data',
            firstname: 'John',
            gender: 'm',
            image: 'data:image/jpg;base64,mock_image_data',
            lastname: 'Doe',
            message: 'Personalization Successful',
            middlename: 'Mock',
            nin: this.generateRandomNIN(),
            signature: 'data:image/jpg;base64,mock_signature_data',
            state: 'SO',
            status: '1',
            town: 'SOKOTO',
            trackingid: request.tracking_id,
        };
    }
    async healthCheck() {
        await this.simulateDelay();
        return !this.shouldSimulateError();
    }
    initializeMockData() {
        this.mockData.set('mock_config', {
            errorRate: 0.1,
            mockDelay: 1000,
        });
    }
    async simulateDelay() {
        const delay = this.config.sandboxMode ? (this.mockData.get('mock_config')?.mockDelay || 1000) : 100;
        await new Promise(resolve => setTimeout(resolve, delay));
    }
    shouldSimulateError() {
        const errorRate = this.mockData.get('mock_config')?.errorRate || 0.1;
        return Math.random() < errorRate;
    }
    generateRandomString(length) {
        const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
        let result = '';
        for (let i = 0; i < length; i++) {
            result += chars.charAt(Math.floor(Math.random() * chars.length));
        }
        return result;
    }
    generateRandomNIN() {
        return Math.floor(10000000000 + Math.random() * 90000000000).toString();
    }
    generateTrackingId() {
        return `S${this.generateRandomString(10)}`;
    }
    validateNinRequest(request) {
        const hasNin = request.nin && request.nin.length === 11;
        const hasNameAndDetails = request.first_name && request.last_name && request.gender && request.dob;
        const hasPhone = request.phone && request.phone.length >= 10;
        if (!hasNin && !hasNameAndDetails && !hasPhone) {
            throw new verification_types_1.ValidationError('NIN verification requires either: NIN (11 digits), name/gender/dob combination, or phone number');
        }
        if (request.nin && !/^\d{11}$/.test(request.nin)) {
            throw new verification_types_1.ValidationError('NIN must be exactly 11 digits');
        }
        if (request.phone && !/^\d{10,11}$/.test(request.phone)) {
            throw new verification_types_1.ValidationError('Phone number must be 10-11 digits');
        }
        if (request.gender && !['male', 'female'].includes(request.gender)) {
            throw new verification_types_1.ValidationError('Gender must be either "male" or "female"');
        }
        if (request.dob && !/^\d{2}-\d{2}-\d{4}$/.test(request.dob)) {
            throw new verification_types_1.ValidationError('Date of birth must be in format: DD-MM-YYYY');
        }
    }
}
exports.MockProvider = MockProvider;
