"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.paystackAdapter = exports.PaystackStub = exports.PaystackIntegration = void 0;
const crypto_1 = __importDefault(require("crypto"));
const axios_1 = __importDefault(require("axios"));
const env_1 = require("../../config/env");
class PaystackIntegration {
    client;
    baseURL = 'https://api.paystack.co';
    constructor() {
        this.client = axios_1.default.create({
            baseURL: this.baseURL,
            headers: {
                'Authorization': `Bearer ${env_1.env.PAYSTACK_SECRET_KEY}`,
                'Content-Type': 'application/json',
            },
            timeout: 30000,
        });
        this.client.interceptors.request.use((config) => {
            console.log(`[Paystack] ${config.method?.toUpperCase()} ${config.url}`);
            return config;
        }, (error) => {
            console.error('[Paystack] Request error:', error);
            return Promise.reject(error);
        });
        this.client.interceptors.response.use((response) => {
            return response;
        }, (error) => {
            console.error('[Paystack] Response error:', error.response?.data || error.message);
            return Promise.reject(error);
        });
    }
    async initializeCharge(request) {
        try {
            const response = await this.client.post('/transaction/initialize', {
                amount: request.amount * 100,
                email: request.email,
                reference: request.reference,
                callback_url: request.callback_url,
                currency: request.currency || 'NGN',
                metadata: request.metadata || {},
                channels: request.channels,
                split_code: request.split_code,
                transaction_charge: request.transaction_charge,
                bearer: request.bearer,
                subaccount: request.subaccount,
                plan: request.plan,
                quantity: request.quantity,
                invoice_limit: request.invoice_limit,
            });
            return response.data;
        }
        catch (error) {
            console.error('[Paystack] Charge initialization failed:', error);
            throw new Error(`Failed to initialize Paystack charge: ${error instanceof Error ? error.message : 'Unknown error'}`);
        }
    }
    async verifyTransaction(reference) {
        try {
            const response = await this.client.get(`/transaction/verify/${reference}`);
            return response.data.data;
        }
        catch (error) {
            console.error('[Paystack] Transaction verification failed:', error);
            throw new Error(`Failed to verify Paystack transaction: ${error instanceof Error ? error.message : 'Unknown error'}`);
        }
    }
    verifyWebhookSignature(signature, rawBody) {
        if (!signature || !env_1.env.PAYSTACK_WEBHOOK_SECRET) {
            console.warn('[Paystack] Missing webhook signature or secret');
            return false;
        }
        try {
            const hash = crypto_1.default
                .createHmac('sha512', env_1.env.PAYSTACK_WEBHOOK_SECRET)
                .update(rawBody, 'utf8')
                .digest('hex');
            return hash === signature;
        }
        catch (error) {
            console.error('[Paystack] Webhook signature verification failed:', error);
            return false;
        }
    }
    async processWebhookEvent(event) {
        try {
            const { data } = event;
            const amount = data.amount / 100;
            return {
                success: data.status === 'success',
                transactionId: data.reference,
                status: data.status,
                amount,
                metadata: data.metadata || {},
            };
        }
        catch (error) {
            console.error('[Paystack] Webhook event processing failed:', error);
            throw new Error(`Failed to process Paystack webhook event: ${error instanceof Error ? error.message : 'Unknown error'}`);
        }
    }
    async reconcileTransactions(startDate, endDate) {
        try {
            const response = await this.client.get('/transaction', {
                params: {
                    from: startDate,
                    to: endDate,
                    perPage: 100,
                },
            });
            return response.data.data;
        }
        catch (error) {
            console.error('[Paystack] Transaction reconciliation failed:', error);
            throw new Error(`Failed to reconcile Paystack transactions: ${error instanceof Error ? error.message : 'Unknown error'}`);
        }
    }
    async refundTransaction(reference, amount) {
        try {
            const response = await this.client.post('/refund', {
                transaction: reference,
                amount: amount ? amount * 100 : undefined,
            });
            return {
                success: response.data.status,
                refundId: response.data.data.id.toString(),
                amount: response.data.data.amount / 100,
            };
        }
        catch (error) {
            console.error('[Paystack] Transaction refund failed:', error);
            throw new Error(`Failed to refund Paystack transaction: ${error instanceof Error ? error.message : 'Unknown error'}`);
        }
    }
    async payBill(input) {
        const chargeResponse = await this.initializeCharge({
            amount: input.amount,
            email: `customer-${input.customerId}@unify.com`,
            reference: input.reference,
            metadata: {
                customerId: input.customerId,
                productCode: input.productCode,
                type: 'bill_payment',
            },
        });
        return {
            success: chargeResponse.status,
            providerReference: chargeResponse.data.reference,
            token: chargeResponse.data.access_code,
            meta: chargeResponse.data,
        };
    }
    async buyVtu(input) {
        const chargeResponse = await this.initializeCharge({
            amount: input.amount,
            email: `vtu-${input.phone}@unify.com`,
            reference: input.reference,
            metadata: {
                type: 'vtu',
                vtuType: input.type,
                network: input.network,
                phone: input.phone,
                planCode: input.planCode,
            },
        });
        return {
            success: chargeResponse.status,
            providerReference: chargeResponse.data.reference,
            meta: chargeResponse.data,
        };
    }
}
exports.PaystackIntegration = PaystackIntegration;
class PaystackStub {
    async initializeCharge(request) {
        return {
            status: true,
            message: 'Charge initialized successfully',
            data: {
                authorization_url: `https://checkout.paystack.com/${request.reference}`,
                access_code: `ACC-${request.reference}`,
                reference: request.reference,
                amount: request.amount * 100,
                currency: request.currency || 'NGN',
                metadata: request.metadata || {},
            },
        };
    }
    async verifyTransaction(reference) {
        return {
            id: 123456789,
            domain: 'test',
            amount: 100000,
            currency: 'NGN',
            source: 'api',
            reason: 'Test transaction',
            recipient: 0,
            status: 'success',
            transfer_code: `TRF-${reference}`,
            created_at: new Date().toISOString(),
            updated_at: new Date().toISOString(),
        };
    }
    verifyWebhookSignature(_signature, _rawBody) {
        return true;
    }
    async processWebhookEvent(event) {
        return {
            success: true,
            transactionId: event.data.reference,
            status: event.data.status,
            amount: event.data.amount / 100,
            metadata: event.data.metadata || {},
        };
    }
    async reconcileTransactions(_startDate, _endDate) {
        return [];
    }
    async refundTransaction(reference, amount) {
        return {
            success: true,
            refundId: `REF-${reference}`,
            amount: amount || 1000,
        };
    }
    async payBill(input) {
        return { success: true, providerReference: `PSK-${input.reference}`, meta: { stub: true } };
    }
    async buyVtu(input) {
        return { success: true, providerReference: `PSK-${input.reference}`, meta: { stub: true } };
    }
}
exports.PaystackStub = PaystackStub;
exports.paystackAdapter = env_1.env.PAYSTACK_SECRET_KEY
    ? new PaystackIntegration()
    : new PaystackStub();
