"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.flutterwaveAdapter = exports.FlutterwaveStub = exports.FlutterwaveIntegration = void 0;
const crypto_1 = __importDefault(require("crypto"));
const axios_1 = __importDefault(require("axios"));
const env_1 = require("../../config/env");
class FlutterwaveIntegration {
    client;
    baseURL = 'https://api.flutterwave.com/v3';
    constructor() {
        this.client = axios_1.default.create({
            baseURL: this.baseURL,
            headers: {
                'Authorization': `Bearer ${env_1.env.FLUTTERWAVE_SECRET_KEY}`,
                'Content-Type': 'application/json',
            },
            timeout: 30000,
        });
        this.client.interceptors.request.use((config) => {
            console.log(`[Flutterwave] ${config.method?.toUpperCase()} ${config.url}`);
            return config;
        }, (error) => {
            console.error('[Flutterwave] Request error:', error);
            return Promise.reject(error);
        });
        this.client.interceptors.response.use((response) => {
            return response;
        }, (error) => {
            console.error('[Flutterwave] Response error:', error.response?.data || error.message);
            return Promise.reject(error);
        });
    }
    async initializeCharge(request) {
        try {
            const response = await this.client.post('/payments', {
                tx_ref: request.tx_ref,
                amount: request.amount,
                currency: request.currency,
                redirect_url: request.redirect_url,
                customer: request.customer,
                customizations: request.customizations,
                meta: request.meta || {},
                payment_options: request.payment_options,
                payment_plan: request.payment_plan,
                subaccounts: request.subaccounts,
                payment_type: request.payment_type,
                is_permanent: request.is_permanent,
                expires: request.expires,
            });
            return response.data;
        }
        catch (error) {
            console.error('[Flutterwave] Charge initialization failed:', error);
            throw new Error(`Failed to initialize Flutterwave charge: ${error instanceof Error ? error.message : 'Unknown error'}`);
        }
    }
    async verifyTransaction(transactionId) {
        try {
            const response = await this.client.get(`/transactions/${transactionId}/verify`);
            return response.data.data;
        }
        catch (error) {
            console.error('[Flutterwave] Transaction verification failed:', error);
            throw new Error(`Failed to verify Flutterwave transaction: ${error instanceof Error ? error.message : 'Unknown error'}`);
        }
    }
    verifyWebhookSignature(signature, rawBody) {
        if (!signature) {
            console.warn('[Flutterwave] Missing webhook signature');
            return false;
        }
        try {
            const hash = crypto_1.default
                .createHmac('sha256', env_1.env.FLUTTERWAVE_SECRET_KEY || '')
                .update(rawBody, 'utf8')
                .digest('hex');
            return hash === signature;
        }
        catch (error) {
            console.error('[Flutterwave] Webhook signature verification failed:', error);
            return false;
        }
    }
    async processWebhookEvent(event) {
        try {
            const { data } = event;
            return {
                success: data.status === 'successful',
                transactionId: data.tx_ref,
                status: data.status,
                amount: data.amount,
                metadata: data.meta || {},
            };
        }
        catch (error) {
            console.error('[Flutterwave] Webhook event processing failed:', error);
            throw new Error(`Failed to process Flutterwave webhook event: ${error instanceof Error ? error.message : 'Unknown error'}`);
        }
    }
    async reconcileTransactions(startDate, endDate) {
        try {
            const response = await this.client.get('/transactions', {
                params: {
                    from: startDate,
                    to: endDate,
                    per_page: 100,
                },
            });
            return response.data.data;
        }
        catch (error) {
            console.error('[Flutterwave] Transaction reconciliation failed:', error);
            throw new Error(`Failed to reconcile Flutterwave transactions: ${error instanceof Error ? error.message : 'Unknown error'}`);
        }
    }
    async refundTransaction(transactionId, amount) {
        try {
            const response = await this.client.post('/refunds', {
                id: transactionId,
                amount: amount,
            });
            return {
                success: response.data.status === 'success',
                refundId: response.data.data.id.toString(),
                amount: response.data.data.amount,
            };
        }
        catch (error) {
            console.error('[Flutterwave] Transaction refund failed:', error);
            throw new Error(`Failed to refund Flutterwave transaction: ${error instanceof Error ? error.message : 'Unknown error'}`);
        }
    }
    async validateBankAccount(accountNumber, _accountBank) {
        try {
            const response = await this.client.post('/accounts/resolve', {
                account_number: accountNumber,
                account_bank: _accountBank,
            });
            return response.data;
        }
        catch (error) {
            console.error('[Flutterwave] Bank account validation failed:', error);
            throw new Error(`Failed to validate bank account: ${error instanceof Error ? error.message : 'Unknown error'}`);
        }
    }
    async payBill(input) {
        const chargeResponse = await this.initializeCharge({
            tx_ref: input.reference,
            amount: input.amount,
            currency: 'NGN',
            customer: {
                email: `customer-${input.customerId}@unify.com`,
                name: `Customer ${input.customerId}`,
            },
            meta: {
                customerId: input.customerId,
                productCode: input.productCode,
                type: 'bill_payment',
            },
        });
        return {
            success: chargeResponse.status === 'success',
            providerReference: chargeResponse.data.flw_ref,
            token: chargeResponse.data.flw_ref,
            meta: chargeResponse.data,
        };
    }
    async buyVtu(input) {
        const chargeResponse = await this.initializeCharge({
            tx_ref: input.reference,
            amount: input.amount,
            currency: 'NGN',
            customer: {
                email: `vtu-${input.phone}@unify.com`,
                phone_number: input.phone,
                name: `VTU Customer`,
            },
            meta: {
                type: 'vtu',
                vtuType: input.type,
                network: input.network,
                phone: input.phone,
                planCode: input.planCode,
            },
        });
        return {
            success: chargeResponse.status === 'success',
            providerReference: chargeResponse.data.flw_ref,
            meta: chargeResponse.data,
        };
    }
}
exports.FlutterwaveIntegration = FlutterwaveIntegration;
class FlutterwaveStub {
    async initializeCharge(request) {
        return {
            status: 'success',
            message: 'Charge initialized successfully',
            data: {
                link: `https://checkout.flutterwave.com/v3/hosted/pay/${request.tx_ref}`,
                tx_ref: request.tx_ref,
                flw_ref: `FLW-${request.tx_ref}`,
                amount: request.amount,
                currency: request.currency,
                meta: request.meta || {},
            },
        };
    }
    async verifyTransaction(transactionId) {
        return {
            id: 123456789,
            tx_ref: transactionId,
            flw_ref: `FLW-${transactionId}`,
            device_fingerprint: 'test-fingerprint',
            amount: 1000,
            currency: 'NGN',
            charged_amount: 1000,
            app_fee: 10,
            merchant_fee: 0,
            processor_response: 'Approved',
            auth_model: 'PIN',
            ip: '127.0.0.1',
            narration: 'Test transaction',
            status: 'successful',
            payment_type: 'card',
            created_at: new Date().toISOString(),
            updated_at: new Date().toISOString(),
            account_id: 1,
            customer: {
                id: 1,
                phone_number: '+2348012345678',
                name: 'Test Customer',
                email: 'test@example.com',
                created_at: new Date().toISOString(),
            },
            meta: {},
        };
    }
    verifyWebhookSignature(_signature, _rawBody) {
        return true;
    }
    async processWebhookEvent(event) {
        return {
            success: true,
            transactionId: event.data.tx_ref,
            status: event.data.status,
            amount: event.data.amount,
            metadata: event.data.meta || {},
        };
    }
    async reconcileTransactions(_startDate, _endDate) {
        return [];
    }
    async refundTransaction(transactionId, amount) {
        return {
            success: true,
            refundId: `REF-${transactionId}`,
            amount: amount || 1000,
        };
    }
    async validateBankAccount(accountNumber, _accountBank) {
        return {
            status: 'success',
            message: 'Account resolved successfully',
            data: {
                account_number: accountNumber,
                account_name: 'Test Account Name',
                bank_id: 1,
            },
        };
    }
    async payBill(input) {
        return { success: true, providerReference: `FLW-${input.reference}`, token: undefined, meta: { stub: true } };
    }
    async buyVtu(input) {
        return { success: true, providerReference: `FLW-${input.reference}`, meta: { stub: true } };
    }
}
exports.FlutterwaveStub = FlutterwaveStub;
exports.flutterwaveAdapter = env_1.env.FLUTTERWAVE_SECRET_KEY
    ? new FlutterwaveIntegration()
    : new FlutterwaveStub();
