"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.configurableServiceManager = exports.ConfigurableServiceManager = void 0;
const service_provider_repo_1 = require("../database/repositories/service-provider.repo");
const vtu_bills_types_1 = require("./vtu-bills/vtu-bills.types");
const asbdata_provider_1 = require("./vtu-bills/asbdata.provider");
const mock_provider_1 = require("./vtu-bills/mock.provider");
const errors_1 = require("../core/errors");
class ConfigurableServiceManager {
    static instance;
    providerCache = new Map();
    planCache = new Map();
    cacheExpiry = new Map();
    CACHE_TTL = 5 * 60 * 1000;
    constructor() {
        setInterval(() => this.cleanupCache(), 60000);
    }
    static getInstance() {
        if (!ConfigurableServiceManager.instance) {
            ConfigurableServiceManager.instance = new ConfigurableServiceManager();
        }
        return ConfigurableServiceManager.instance;
    }
    async getProvider(serviceType, fallbackOnFailure = true) {
        const cacheKey = `provider_${serviceType}`;
        if (this.providerCache.has(cacheKey) && this.isCacheValid(cacheKey)) {
            return this.providerCache.get(cacheKey);
        }
        const providers = await service_provider_repo_1.serviceProviderRepository.findByPriority(serviceType, 5);
        if (providers.length === 0) {
            throw new errors_1.AppError(`No active providers found for service type: ${serviceType}`, 503);
        }
        for (const providerConfig of providers) {
            try {
                const adapter = await this.createProviderAdapter(providerConfig);
                const healthStatus = await adapter.checkHealth();
                if (healthStatus.status === 'online') {
                    this.providerCache.set(cacheKey, adapter);
                    this.cacheExpiry.set(cacheKey, Date.now() + this.CACHE_TTL);
                    return adapter;
                }
            }
            catch (error) {
                console.warn(`Provider ${providerConfig.name} failed health check:`, error);
                await service_provider_repo_1.serviceProviderRepository.incrementErrorCount(providerConfig.id);
                if (!fallbackOnFailure) {
                    throw error;
                }
            }
        }
        throw new errors_1.AppError(`All providers for ${serviceType} are currently unavailable`, 503);
    }
    async getServicePlans(type, network, filters = {}) {
        const cacheKey = `plans_${type}_${network || 'all'}_${JSON.stringify(filters)}`;
        if (this.planCache.has(cacheKey) && this.isCacheValid(cacheKey)) {
            return this.planCache.get(cacheKey);
        }
        const { ProductRepository } = await Promise.resolve().then(() => __importStar(require('../database/repositories/product.repo')));
        const productRepo = new ProductRepository();
        let products = [];
        if (type === 'data' && network) {
            products = await productRepo.findActiveByNetworkAndType(network, 'data', filters.dataPlanType);
        }
        else if (type === 'airtime' && network) {
            products = await productRepo.findActiveByNetworkAndType(network, 'airtime');
        }
        else {
            products = [];
        }
        const mappedPlans = products.map(product => ({
            id: product._id,
            planId: product.canonicalCode,
            vendorPlanId: product.providers?.[0]?.vendorCode || product.canonicalCode,
            name: product.name,
            type: product.type,
            network: product.network,
            size: product.size,
            validity: product.validity,
            amount: product.providers?.[0]?.costPrice || 0,
            sellingPrice: product.sellingPrice,
            status: 'active',
            isVisible: product.isVisible,
            stockQuantity: null,
            metadata: product.metadata || {}
        }));
        this.planCache.set(cacheKey, mappedPlans);
        this.cacheExpiry.set(cacheKey, Date.now() + this.CACHE_TTL);
        return mappedPlans;
    }
    async executeVtuPurchase(request) {
        const serviceType = 'vtu';
        let lastError = null;
        const providers = await service_provider_repo_1.serviceProviderRepository.findByPriority(serviceType, 3);
        for (const providerConfig of providers) {
            try {
                const adapter = await this.createProviderAdapter(providerConfig);
                const result = request.type === 'airtime'
                    ? await adapter.buyAirtime(request)
                    : await adapter.buyData(request);
                if (result.success) {
                    await service_provider_repo_1.serviceProviderRepository.resetErrorCount(providerConfig.id);
                }
                else {
                    await service_provider_repo_1.serviceProviderRepository.incrementErrorCount(providerConfig.id);
                }
                return result;
            }
            catch (error) {
                console.error(`VTU purchase failed with provider ${providerConfig.name}:`, error);
                lastError = error;
                const isProviderLowBalance = error?.code === 'INSUFFICIENT_PROVIDER_BALANCE' ||
                    (error instanceof Error && error?.code === 'INSUFFICIENT_PROVIDER_BALANCE') ||
                    (error instanceof vtu_bills_types_1.VtuBillsError && error.code === 'INSUFFICIENT_PROVIDER_BALANCE');
                if (isProviderLowBalance) {
                    console.warn(`[ProviderFallback] ${providerConfig.name} reports insufficient provider balance. Trying next provider...`);
                    console.log(JSON.stringify({
                        event: 'provider_low_balance',
                        provider: providerConfig.name,
                        serviceType,
                        timestamp: new Date().toISOString(),
                    }));
                }
                else {
                    await service_provider_repo_1.serviceProviderRepository.incrementErrorCount(providerConfig.id);
                }
            }
        }
        throw new errors_1.AppError(`All VTU providers failed. Last error: ${lastError?.message || 'Unknown error'}`, 503);
    }
    async executeBillPayment(request) {
        const serviceType = 'bills';
        let lastError = null;
        const providers = await service_provider_repo_1.serviceProviderRepository.findByPriority(serviceType, 3);
        for (const providerConfig of providers) {
            try {
                const adapter = await this.createProviderAdapter(providerConfig);
                let result;
                switch (request.type) {
                    case 'electricity':
                        result = await adapter.payElectricityBill(request);
                        break;
                    case 'cable':
                        result = await adapter.payCableBill(request);
                        break;
                    case 'education':
                        result = await adapter.payEducationBill(request);
                        break;
                    default:
                        throw new errors_1.AppError(`Unsupported bill type: ${request.type}`, 400);
                }
                if (result.success) {
                    await service_provider_repo_1.serviceProviderRepository.resetErrorCount(providerConfig.id);
                }
                else {
                    await service_provider_repo_1.serviceProviderRepository.incrementErrorCount(providerConfig.id);
                }
                return result;
            }
            catch (error) {
                console.error(`Bill payment failed with provider ${providerConfig.name}:`, error);
                lastError = error;
                const isProviderLowBalance = error?.code === 'INSUFFICIENT_PROVIDER_BALANCE' ||
                    (error instanceof Error && error?.code === 'INSUFFICIENT_PROVIDER_BALANCE') ||
                    (error instanceof vtu_bills_types_1.VtuBillsError && error.code === 'INSUFFICIENT_PROVIDER_BALANCE');
                if (isProviderLowBalance) {
                    console.warn(`[ProviderFallback] ${providerConfig.name} reports insufficient provider balance. Trying next provider...`);
                    console.log(JSON.stringify({
                        event: 'provider_low_balance',
                        provider: providerConfig.name,
                        serviceType,
                        timestamp: new Date().toISOString(),
                    }));
                }
                else {
                    await service_provider_repo_1.serviceProviderRepository.incrementErrorCount(providerConfig.id);
                }
            }
        }
        throw new errors_1.AppError(`All bill payment providers failed. Last error: ${lastError?.message || 'Unknown error'}`, 503);
    }
    async getPlanById(planId, providerId) {
        const { ProductRepository } = await Promise.resolve().then(() => __importStar(require('../database/repositories/product.repo')));
        const productRepo = new ProductRepository();
        const product = await productRepo.findByCanonicalCode(planId);
        if (!product)
            return null;
        return {
            id: product._id,
            planId: product.canonicalCode,
            vendorPlanId: product.providers?.[0]?.vendorCode || product.canonicalCode,
            name: product.name,
            type: product.type,
            network: product.network,
            size: product.size,
            validity: product.validity,
            amount: product.providers?.[0]?.costPrice || 0,
            sellingPrice: product.sellingPrice,
            status: 'active',
            isVisible: product.isVisible,
            stockQuantity: null,
            metadata: product.metadata || {}
        };
    }
    async calculatePrice(planId, providerId) {
        const plan = await this.getPlanById(planId, providerId);
        if (!plan) {
            throw new errors_1.AppError('Plan not found', 404);
        }
        const provider = await service_provider_repo_1.serviceProviderRepository.findById(plan.providerId.toString());
        if (!provider) {
            throw new errors_1.AppError('Provider not found', 404);
        }
        const markup = provider.defaultMarkup || 0;
        const basePrice = plan.amount;
        const sellingPrice = plan.sellingPrice;
        return {
            basePrice,
            sellingPrice,
            markup,
        };
    }
    async validateServiceAvailability(serviceType, service) {
        const providers = await service_provider_repo_1.serviceProviderRepository.findBySupportedService(service, serviceType);
        return providers.length > 0;
    }
    async getServiceStats() {
        const providerStats = await service_provider_repo_1.serviceProviderRepository.getProviderStats();
        const planStats = {};
        const allProviders = await service_provider_repo_1.serviceProviderRepository.find({ status: 'active' });
        const healthStatus = {};
        for (const provider of allProviders) {
            healthStatus[provider.slug] = {
                isHealthy: provider.isHealthy,
                lastCheck: provider.lastHealthCheck,
                errorCount: provider.errorCount,
                successRate: provider.successRate,
            };
        }
        return {
            providers: providerStats,
            plans: planStats,
            healthStatus,
        };
    }
    clearCache(key) {
        if (key) {
            this.providerCache.delete(key);
            this.planCache.delete(key);
            this.cacheExpiry.delete(key);
        }
        else {
            this.providerCache.clear();
            this.planCache.clear();
            this.cacheExpiry.clear();
        }
    }
    async createProviderAdapter(providerConfig) {
        let baseAdapter;
        const decryptedConfig = providerConfig.decryptFields(['apiKey', 'secretKey']);
        switch (providerConfig.slug) {
            case 'asbdata':
                baseAdapter = new asbdata_provider_1.ASBDataProvider({
                    name: providerConfig.displayName,
                    baseUrl: providerConfig.baseUrl,
                    apiKey: decryptedConfig.apiKey || providerConfig.apiKey,
                    secretKey: decryptedConfig.secretKey || providerConfig.secretKey,
                    timeout: providerConfig.timeout,
                    retryAttempts: providerConfig.retryAttempts,
                    sandboxMode: providerConfig.sandboxMode,
                    enabled: providerConfig.status === 'active',
                });
                break;
            case 'mock':
                baseAdapter = new mock_provider_1.MockProvider({
                    name: providerConfig.displayName,
                    baseUrl: providerConfig.baseUrl,
                    apiKey: decryptedConfig.apiKey || providerConfig.apiKey,
                    timeout: providerConfig.timeout,
                    retryAttempts: providerConfig.retryAttempts,
                    sandboxMode: true,
                    enabled: providerConfig.status === 'active',
                });
                break;
            default:
                throw new errors_1.AppError(`Unsupported provider: ${providerConfig.slug}`, 500);
        }
        return {
            ...baseAdapter,
            providerId: providerConfig.id,
            providerConfig,
        };
    }
    isCacheValid(key) {
        const expiry = this.cacheExpiry.get(key);
        return expiry ? Date.now() < expiry : false;
    }
    cleanupCache() {
        const now = Date.now();
        for (const [key, expiry] of this.cacheExpiry.entries()) {
            if (now >= expiry) {
                this.providerCache.delete(key);
                this.planCache.delete(key);
                this.cacheExpiry.delete(key);
            }
        }
    }
}
exports.ConfigurableServiceManager = ConfigurableServiceManager;
exports.configurableServiceManager = ConfigurableServiceManager.getInstance();
