"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.setAspfiyAdapterMode = exports.getAspfiyAdapterMode = exports.aspfiyAdapter = exports.AspfiyIntegration = void 0;
const axios_1 = __importDefault(require("axios"));
const crypto_1 = __importDefault(require("crypto"));
const env_1 = require("../../config/env");
const errors_1 = require("../../core/errors");
const utils_1 = require("./utils");
Object.defineProperty(exports, "getAspfiyAdapterMode", { enumerable: true, get: function () { return utils_1.getAspfiyAdapterMode; } });
Object.defineProperty(exports, "setAspfiyAdapterMode", { enumerable: true, get: function () { return utils_1.setAspfiyAdapterMode; } });
class AspfiyIntegration {
    client;
    mode;
    endpoints = {
        reservedAccount: '/reserve-account/',
        reservePaga: '/reserve-paga/',
        reservePalmpay: '/reserve-palmpay/',
        dynamicAccount: '/reserve-dynamic-account/',
        banks: '/transfers/banks/',
        resolveAccount: '/transfers/resolve-nuban/',
        transfer: '/transfer/initiate/',
        payment: '/payment/initialize/',
        refund: '/refund/',
        transactionVerify: (ref) => `/transaction/verify/${ref}`,
    };
    constructor() {
        this.mode = (0, utils_1.getAspfiyAdapterMode)();
        this.client = axios_1.default.create({
            baseURL: env_1.env.ASPFIY_BASE_URL || '',
            headers: {
                Authorization: `Bearer ${env_1.env.ASPFIY_SECRET_KEY}`,
                'Content-Type': 'application/json',
            },
        });
        this.client.interceptors.response.use((res) => {
            console.log('[Aspfiy Response]', res.config.url, res.status, res.data);
            return res;
        }, (err) => {
            console.error('[Aspfiy Error]', err.response?.status, err.response?.data || err.message);
            return Promise.reject(err);
        });
    }
    handleError(error, context) {
        const status = error?.response?.status;
        const message = error?.response?.data?.message || error?.message || 'Unknown error';
        console.error(`[Aspfiy] ${context} failed:`, status, message);
        throw new errors_1.ExternalServiceError(`${context} failed: ${message}`);
    }
    async chargeCustomer(request) {
        if (this.mode === 'stub') {
            return {
                status: true,
                message: 'Stub charge',
                data: {
                    reference: request.reference,
                    amount: request.amount,
                    currency: request.currency || 'NGN',
                    authorization_url: 'https://stub.aspfiy/authorize',
                    metadata: request.metadata || {},
                },
            };
        }
        try {
            const res = await this.client.post(this.endpoints.payment, request);
            return res.data;
        }
        catch (err) {
            this.handleError(err, 'Charge customer');
        }
    }
    async createReservedAccount(request) {
        if (this.mode === 'stub') {
            return {
                status: true,
                message: 'Stub reserved account',
                data: {
                    account_number: '0001234567',
                    account_name: 'John Doe',
                    bank_name: 'Stub Bank',
                    bank_code: '001',
                    reference: request.reference,
                    customer_email: request.customer_email,
                    created_at: new Date().toISOString(),
                    metadata: request.metadata || {},
                },
            };
        }
        try {
            const path = request.provider === 'paga'
                ? this.endpoints.reservePaga
                : request.provider === 'palmpay'
                    ? this.endpoints.reservePalmpay
                    : this.endpoints.reservedAccount;
            const fullName = (request.customer_name || '').trim();
            const nameParts = fullName.split(/\s+/).filter(Boolean);
            let firstname = request.firstname || nameParts[0] || '';
            let lastname = request.lastname || (nameParts.length > 1 ? nameParts.slice(1).join(' ') : '');
            if (!firstname || firstname.trim().length === 0) {
                const local = (request.customer_email || '').split('@')[0] || '';
                const tokens = local.split(/[._-]+/).filter(Boolean);
                firstname = tokens[0] || 'User';
                if (!lastname || lastname.trim().length === 0) {
                    lastname = tokens[1] || 'Account';
                }
            }
            const rawPhone = (request.customer_phone || '').toString();
            const onlyDigits = rawPhone.replace(/\D/g, '');
            let phone11 = onlyDigits;
            if (onlyDigits.startsWith('234') && onlyDigits.length >= 13) {
                phone11 = '0' + onlyDigits.slice(3, 13);
            }
            else if (onlyDigits.length === 10) {
                phone11 = '0' + onlyDigits;
            }
            else if (onlyDigits.length > 11 && onlyDigits.endsWith('11')) {
                phone11 = onlyDigits.slice(-11);
            }
            if (phone11.length !== 11) {
                phone11 = onlyDigits;
            }
            const payload = {
                email: request.customer_email,
                reference: request['reference'] || request.reference,
                firstName: firstname,
                lastName: lastname,
                phone: phone11,
                webhookUrl: `${env_1.env.APP_URL}/api/v1/webhooks/aspfiy`,
            };
            const reqAny = request;
            if (Object.prototype.hasOwnProperty.call(reqAny, 'reference') && reqAny['reference']) {
                payload['reference'] = reqAny['reference'];
            }
            if (Object.prototype.hasOwnProperty.call(reqAny, 'metadata') && reqAny['metadata']) {
                payload['metadata'] = reqAny['metadata'];
            }
            const res = await this.client.post(path, payload);
            return res.data;
        }
        catch (err) {
            this.handleError(err, 'Create reserved account');
        }
    }
    async createDynamicAccount(request) {
        if (this.mode === 'stub') {
            return {
                status: true,
                message: 'Stub dynamic account',
                data: {
                    account_number: '0009876543',
                    account_name: 'Jane Doe',
                    bank_name: 'Dynamic Stub Bank',
                    reference: request.reference,
                    amount: request.amount,
                    currency: 'NGN',
                    expires_at: new Date(Date.now() + (request.expires_in || 60) * 60000).toISOString(),
                    customer_email: request.customer_email,
                    created_at: new Date().toISOString(),
                    metadata: request.metadata || {},
                },
            };
        }
        try {
            const res = await this.client.post(this.endpoints.dynamicAccount, request);
            return res.data;
        }
        catch (err) {
            this.handleError(err, 'Create dynamic account');
        }
    }
    async initiateTransfer(request) {
        if (this.mode === 'stub') {
            return {
                status: true,
                message: 'Stub transfer',
                data: {
                    reference: request.reference,
                    amount: request.amount,
                    currency: 'NGN',
                    recipient_account: request.recipient_account,
                    recipient_bank_code: request.recipient_bank_code,
                    ...(request.recipient_name ? { recipient_name: request.recipient_name } : {}),
                    ...(request.narration ? { narration: request.narration } : {}),
                    status: 'pending',
                    created_at: new Date().toISOString(),
                    updated_at: new Date().toISOString(),
                    metadata: request.metadata || {},
                },
            };
        }
        try {
            const res = await this.client.post(this.endpoints.transfer, request);
            return res.data;
        }
        catch (err) {
            this.handleError(err, 'Initiate transfer');
        }
    }
    async getBanks() {
        if (this.mode === 'stub') {
            return {
                status: true,
                message: 'Stub banks',
                data: [
                    { id: 1, name: 'Stub Bank 1', code: '001', active: true },
                    { id: 2, name: 'Stub Bank 2', code: '002', active: true },
                ],
            };
        }
        try {
            const res = await this.client.get(this.endpoints.banks);
            return res.data;
        }
        catch (err) {
            this.handleError(err, 'Get banks');
        }
    }
    async resolveAccount(bankCode, accountNumber) {
        if (this.mode === 'stub') {
            return {
                status: true,
                message: 'Stub account resolution',
                data: {
                    account_number: accountNumber,
                    account_name: 'John Doe',
                    bank_code: bankCode,
                },
            };
        }
        try {
            const res = await this.client.post(this.endpoints.resolveAccount, {
                bank_code: bankCode,
                account_number: accountNumber,
            });
            return res.data;
        }
        catch (err) {
            this.handleError(err, 'Resolve account');
        }
    }
    async refundTransaction(request) {
        if (this.mode === 'stub') {
            return {
                status: true,
                message: 'Stub refund',
                data: {
                    id: 'stub-refund-id',
                    transaction: request.transaction,
                    amount: request.amount || 0,
                    currency: 'NGN',
                    status: 'processed',
                    refunded_at: new Date().toISOString(),
                    created_at: new Date().toISOString(),
                },
            };
        }
        try {
            const res = await this.client.post(this.endpoints.refund, request);
            return res.data;
        }
        catch (err) {
            this.handleError(err, 'Refund transaction');
        }
    }
    async verifyTransaction(reference) {
        if (this.mode === 'stub') {
            return {
                status: true,
                message: 'Stub transaction verify',
                data: {
                    id: 'stub-id',
                    reference,
                    amount: 1000,
                    currency: 'NGN',
                    status: 'success',
                    customer: { id: 'cust-001', email: 'test@stub.com', name: 'John Doe' },
                    created_at: new Date().toISOString(),
                },
            };
        }
        try {
            const res = await this.client.get(this.endpoints.transactionVerify(reference));
            return res.data;
        }
        catch (err) {
            this.handleError(err, 'Verify transaction');
        }
    }
    verifyWebhookSignature(payload, signature) {
        if (!env_1.env.ASPFIY_WEBHOOK_SECRET) {
            console.warn('[Aspfiy] Webhook secret not configured, skipping verification');
            return true;
        }
        if (!signature) {
            console.error('[Aspfiy] No signature provided for webhook verification');
            return false;
        }
        try {
            const expectedHash = crypto_1.default.createHmac('sha256', env_1.env.ASPFIY_WEBHOOK_SECRET).update(payload).digest('hex');
            const signatureFormats = [
                signature,
                signature.replace(/^sha256=/, ''),
                signature.toLowerCase(),
                signature.toUpperCase(),
            ];
            for (const sigFormat of signatureFormats) {
                try {
                    if (sigFormat.length === expectedHash.length && /^[0-9a-fA-F]+$/.test(sigFormat)) {
                        const hashBuffer = Buffer.from(expectedHash, 'hex');
                        const sigBuffer = Buffer.from(sigFormat, 'hex');
                        if (hashBuffer.length === sigBuffer.length && crypto_1.default.timingSafeEqual(hashBuffer, sigBuffer)) {
                            console.log('[Aspfiy] Webhook signature verified successfully');
                            return true;
                        }
                    }
                    if (expectedHash.toLowerCase() === sigFormat.toLowerCase()) {
                        console.log('[Aspfiy] Webhook signature verified successfully (string comparison)');
                        return true;
                    }
                }
                catch (err) {
                    continue;
                }
            }
            console.error('[Aspfiy] Webhook signature verification failed for all formats');
            console.error('[Aspfiy] Expected:', expectedHash);
            console.error('[Aspfiy] Received:', signature);
            return false;
        }
        catch (error) {
            console.error('[Aspfiy] Webhook signature verification error:', error);
            return false;
        }
    }
    handleWebhook(event) {
        console.log('[Aspfiy Webhook]', event.event, event.data);
        return event;
    }
}
exports.AspfiyIntegration = AspfiyIntegration;
const integration = new AspfiyIntegration();
exports.aspfiyAdapter = {
    async initializeCharge(request) {
        const resp = await integration.chargeCustomer(request);
        return resp;
    },
    async verifyTransaction(reference) {
        const resp = await integration.verifyTransaction(reference);
        const data = resp.data;
        return {
            id: data.id,
            reference: data.reference,
            amount: data.amount,
            currency: data.currency,
            status: (data.status === 'success' ? 'successful' : data.status),
            payment_type: data.payment_type || 'bank_transfer',
            account_number: data.account_number,
            account_name: data.account_name,
            bank_name: data.bank_name,
            customer_email: data.customer.email,
            customer_name: data.customer.name,
            customer_phone: data.customer.phone,
            created_at: data.created_at,
            updated_at: data.updated_at || data.created_at,
            metadata: data.metadata,
            fees: data.fees,
            net_amount: data.net_amount,
        };
    },
    verifyWebhookSignature(signature, rawBody) {
        return integration.verifyWebhookSignature(rawBody, signature || '');
    },
    async processWebhookEvent(event) {
        const handled = integration.handleWebhook(event);
        const tx = handled.data;
        const isSuccessful = tx.type === 'RESERVED_ACCOUNT_TRANSACTION' ||
            tx.status === 'successful' ||
            tx.status === 'success';
        const uniqueId = tx.wiaxy_ref || tx.transaction_ref || tx.reference;
        return {
            success: true,
            transactionId: uniqueId,
            status: isSuccessful ? 'successful' : 'pending',
            amount: tx.amount,
            metadata: {
                ...tx.metadata,
                account_number: tx.account?.account_number || tx.account_number,
                account_name: tx.account?.account_name || tx.account_name,
                bank_name: tx.account?.bank_name || tx.bank_name,
                customer_email: tx.customer?.email || tx.customer_email,
                customer_name: tx.customer?.firstName && tx.customer?.lastName
                    ? `${tx.customer.firstName} ${tx.customer.lastName}`
                    : tx.customer_name,
                payment_type: 'reserved_account',
                wiaxy_ref: tx.wiaxy_ref,
                transaction_ref: tx.transaction_ref,
                merchant_reference: tx.merchant_reference,
                created_at: tx.created_at,
                external_reference: tx.reference,
            },
        };
    },
    async reconcileTransactions(_startDate, _endDate) {
        return [];
    },
    async refundTransaction(reference, amount) {
        const req = { transaction: reference };
        if (typeof amount === 'number')
            req.amount = amount;
        const resp = await integration.refundTransaction(req);
        return {
            success: resp.status,
            refundId: resp.data.id,
            amount: resp.data.amount,
        };
    },
    async createReservedAccount(request) {
        return integration.createReservedAccount(request);
    },
    async createDynamicAccount(request) {
        return integration.createDynamicAccount(request);
    },
    async getBanks() {
        const resp = await integration.getBanks();
        return resp.data;
    },
    async resolveAccount(request) {
        const resp = await integration.resolveAccount(request.bank_code, request.account_number);
        return {
            status: resp.status,
            message: resp.message,
            data: {
                account_number: resp.data.account_number,
                account_name: resp.data.account_name,
                bank_id: 0,
                bank_name: resp.data.bank_code,
            },
        };
    },
    async initiateTransfer(request) {
        return integration.initiateTransfer(request);
    },
};
