"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createAggregationPipeline = exports.sanitizeQuery = exports.createNumericRangeFilter = exports.createDateRangeFilter = exports.createTextSearch = exports.applyPagination = exports.createPagination = exports.createSort = exports.createFilter = exports.toJSON = exports.toPlainObject = exports.generateObjectId = exports.isValidObjectId = exports.stringToObjectId = exports.objectIdToString = void 0;
const mongoose_1 = __importDefault(require("mongoose"));
const objectIdToString = (id) => {
    return id.toString();
};
exports.objectIdToString = objectIdToString;
const stringToObjectId = (id) => {
    return new mongoose_1.default.Types.ObjectId(id);
};
exports.stringToObjectId = stringToObjectId;
const isValidObjectId = (id) => {
    return mongoose_1.default.Types.ObjectId.isValid(id);
};
exports.isValidObjectId = isValidObjectId;
const generateObjectId = () => {
    return new mongoose_1.default.Types.ObjectId();
};
exports.generateObjectId = generateObjectId;
const toPlainObject = (doc) => {
    if (doc && typeof doc === 'object' && 'toObject' in doc) {
        return doc.toObject();
    }
    return doc;
};
exports.toPlainObject = toPlainObject;
const toJSON = (doc) => {
    if (doc && typeof doc === 'object' && 'toJSON' in doc) {
        return doc.toJSON();
    }
    return doc;
};
exports.toJSON = toJSON;
const createFilter = (filter) => {
    const mongoFilter = {};
    for (const [key, value] of Object.entries(filter)) {
        if (value !== undefined && value !== null) {
            if (key === '_id' && typeof value === 'string') {
                mongoFilter[key] = (0, exports.stringToObjectId)(value);
            }
            else {
                mongoFilter[key] = value;
            }
        }
    }
    return mongoFilter;
};
exports.createFilter = createFilter;
const createSort = (sort) => {
    return sort;
};
exports.createSort = createSort;
const createPagination = (page = 1, limit = 10) => {
    const skip = (page - 1) * limit;
    return { skip, limit, page };
};
exports.createPagination = createPagination;
const applyPagination = (query, page = 1, limit = 10) => {
    const { skip } = (0, exports.createPagination)(page, limit);
    return query.skip(skip).limit(limit);
};
exports.applyPagination = applyPagination;
const createTextSearch = (searchTerm, fields) => {
    if (!searchTerm || !fields.length)
        return {};
    const searchConditions = fields.map(field => ({
        [field]: { $regex: searchTerm, $options: 'i' }
    }));
    return { $or: searchConditions };
};
exports.createTextSearch = createTextSearch;
const createDateRangeFilter = (startDate, endDate, field = 'createdAt') => {
    const filter = {};
    if (startDate || endDate) {
        filter[field] = {};
        if (startDate)
            filter[field].$gte = startDate;
        if (endDate)
            filter[field].$lte = endDate;
    }
    return filter;
};
exports.createDateRangeFilter = createDateRangeFilter;
const createNumericRangeFilter = (field, min, max) => {
    const filter = {};
    if (min !== undefined || max !== undefined) {
        filter[field] = {};
        if (min !== undefined)
            filter[field].$gte = min;
        if (max !== undefined)
            filter[field].$lte = max;
    }
    return filter;
};
exports.createNumericRangeFilter = createNumericRangeFilter;
const sanitizeQuery = (query) => {
    const sanitized = {};
    for (const [key, value] of Object.entries(query)) {
        if (value !== undefined && value !== null && value !== '') {
            if (key === 'page' || key === 'limit') {
                const numValue = parseInt(value, 10);
                if (!isNaN(numValue) && numValue > 0) {
                    sanitized[key] = numValue;
                }
            }
            else if (key === 'sort') {
                if (typeof value === 'string') {
                    const sortObj = {};
                    const sortFields = value.split(',');
                    for (const field of sortFields) {
                        const trimmedField = field.trim();
                        if (trimmedField.startsWith('-')) {
                            sortObj[trimmedField.slice(1)] = -1;
                        }
                        else {
                            sortObj[trimmedField] = 1;
                        }
                    }
                    if (Object.keys(sortObj).length > 0) {
                        sanitized[key] = sortObj;
                    }
                }
            }
            else {
                sanitized[key] = value;
            }
        }
    }
    return sanitized;
};
exports.sanitizeQuery = sanitizeQuery;
const createAggregationPipeline = (options) => {
    const pipeline = [];
    if (options.match) {
        pipeline.push({ $match: options.match });
    }
    if (options.lookup) {
        for (const lookup of options.lookup) {
            pipeline.push({
                $lookup: {
                    from: lookup.from,
                    localField: lookup.localField,
                    foreignField: lookup.foreignField,
                    as: lookup.as
                }
            });
        }
    }
    if (options.sort) {
        pipeline.push({ $sort: options.sort });
    }
    if (options.group) {
        pipeline.push({ $group: options.group });
    }
    if (options.project) {
        pipeline.push({ $project: options.project });
    }
    if (options.skip) {
        pipeline.push({ $skip: options.skip });
    }
    if (options.limit) {
        pipeline.push({ $limit: options.limit });
    }
    return pipeline;
};
exports.createAggregationPipeline = createAggregationPipeline;
