#!/usr/bin/env ts-node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const connection_1 = require("./connection");
const logger_1 = require("../config/logger");
async function main() {
    try {
        logger_1.logger.info('Starting database connection test...');
        logger_1.logger.info('=== Test 1: Initial Status ===');
        const initialStatus = (0, connection_1.getDatabaseStatus)();
        const isConnected = (0, connection_1.isDatabaseConnected)();
        logger_1.logger.info(`Initial status: ${initialStatus}`);
        logger_1.logger.info(`Is connected: ${isConnected}`);
        logger_1.logger.info('\n=== Test 2: Connection Test ===');
        const connectionSuccess = await (0, connection_1.testConnection)();
        logger_1.logger.info(`Connection test result: ${connectionSuccess ? 'SUCCESS' : 'FAILED'}`);
        if (!connectionSuccess) {
            logger_1.logger.error('Connection test failed, exiting...');
            process.exit(1);
        }
        logger_1.logger.info('\n=== Test 3: Health Check ===');
        const healthCheck = await (0, connection_1.performHealthCheck)();
        logger_1.logger.info(`Health check result: ${healthCheck ? 'HEALTHY' : 'UNHEALTHY'}`);
        logger_1.logger.info('\n=== Test 4: Database Stats ===');
        try {
            const stats = await (0, connection_1.getDatabaseStats)();
            logger_1.logger.info('Database statistics:');
            Object.entries(stats).forEach(([key, value]) => {
                logger_1.logger.info(`  ${key}: ${value}`);
            });
        }
        catch (error) {
            logger_1.logger.warn('Could not retrieve database stats:', error);
        }
        logger_1.logger.info('\n=== Test 5: Final Status ===');
        const finalStatus = (0, connection_1.getDatabaseStatus)();
        const finalConnected = (0, connection_1.isDatabaseConnected)();
        logger_1.logger.info(`Final status: ${finalStatus}`);
        logger_1.logger.info(`Final connected: ${finalConnected}`);
        logger_1.logger.info('\n=== All Tests Completed Successfully ===');
    }
    catch (error) {
        logger_1.logger.error('Test failed with error:', error);
        process.exit(1);
    }
    finally {
        try {
            await (0, connection_1.closeDatabase)();
            logger_1.logger.info('Database connection closed');
        }
        catch (error) {
            logger_1.logger.warn('Error closing database connection:', error);
        }
        process.exit(0);
    }
}
process.on('SIGINT', async () => {
    logger_1.logger.info('Received SIGINT, shutting down gracefully...');
    try {
        await (0, connection_1.closeDatabase)();
        logger_1.logger.info('Graceful shutdown completed');
    }
    catch (error) {
        logger_1.logger.error('Error during graceful shutdown:', error);
    }
    process.exit(0);
});
process.on('SIGTERM', async () => {
    logger_1.logger.info('Received SIGTERM, shutting down gracefully...');
    try {
        await (0, connection_1.closeDatabase)();
        logger_1.logger.info('Graceful shutdown completed');
    }
    catch (error) {
        logger_1.logger.error('Error during graceful shutdown:', error);
    }
    process.exit(0);
});
if (require.main === module) {
    main().catch((error) => {
        logger_1.logger.error('Unhandled error in main:', error);
        process.exit(1);
    });
}
