"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.transactionRepository = exports.TransactionRepository = void 0;
const mongoose_1 = __importDefault(require("mongoose"));
const base_1 = require("./base");
const transactions_1 = require("../models/transactions");
class TransactionRepository extends base_1.BaseRepository {
    model = transactions_1.Transaction;
    isValidId(id) {
        return mongoose_1.default.Types.ObjectId.isValid(id);
    }
    async findByReference(reference) {
        return this.model.findOne({ reference });
    }
    async markStatus(id, status) {
        if (!this.isValidId(id))
            return;
        await this.model.updateOne({ _id: id }, { $set: { status } });
    }
    async getJournalEntry(journalEntryId) {
        return this.model.find({ journalEntryId }).sort({ createdAt: 1 });
    }
    async getRelatedTransactions(transactionId) {
        return this.model.find({
            $or: [
                { _id: transactionId },
                { relatedTransactionId: new mongoose_1.default.Types.ObjectId(transactionId) }
            ]
        }).sort({ createdAt: 1 });
    }
    async exportTransactions(params) {
        const filter = {};
        if (params.userId)
            filter.userId = new mongoose_1.default.Types.ObjectId(params.userId);
        if (params.walletId)
            filter.walletId = new mongoose_1.default.Types.ObjectId(params.walletId);
        if (params.type)
            filter.type = params.type;
        if (params.category)
            filter.category = params.category;
        if (params.status)
            filter.status = params.status;
        if (params.currency)
            filter.currency = params.currency;
        if (params.dateFrom || params.dateTo) {
            filter.createdAt = {};
            if (params.dateFrom)
                filter.createdAt.$gte = params.dateFrom;
            if (params.dateTo)
                filter.createdAt.$lte = params.dateTo;
        }
        if (params.minAmount || params.maxAmount) {
            filter.amount = {};
            if (params.minAmount)
                filter.amount.$gte = params.minAmount;
            if (params.maxAmount)
                filter.amount.$lte = params.maxAmount;
        }
        const transactions = await this.model.find(filter).sort({ createdAt: -1 });
        if (params.format === 'csv') {
            return this.toCSV(transactions, params.includeMetadata);
        }
        else {
            return JSON.stringify(transactions, null, 2);
        }
    }
    toCSV(transactions, includeMetadata = false) {
        const headers = [
            'ID', 'Reference', 'Type', 'Category', 'Amount', 'Currency', 'Status',
            'UserId', 'WalletId', 'BalanceBefore', 'BalanceAfter', 'FeeAmount', 'VATAmount',
            'NetAmount', 'JournalEntryId', 'ExternalReference', 'CreatedAt', 'UpdatedAt'
        ];
        if (includeMetadata) {
            headers.push('Metadata');
        }
        const csvRows = [headers.join(',')];
        for (const tx of transactions) {
            const row = [
                tx.id,
                tx.reference,
                tx.type,
                tx.category,
                tx.amount,
                tx.currency,
                tx.status,
                tx.userId,
                tx.walletId,
                tx.balanceBefore || '',
                tx.balanceAfter || '',
                tx.feeAmount || '',
                tx.vatAmount || '',
                tx.netAmount || '',
                tx.journalEntryId || '',
                tx.externalReference || '',
                tx.createdAt.toISOString(),
                tx.updatedAt.toISOString()
            ];
            if (includeMetadata) {
                row.push(JSON.stringify(tx.metadata || {}));
            }
            csvRows.push(row.map(field => `"${field}"`).join(','));
        }
        return csvRows.join('\n');
    }
    async getTransactionStats(params) {
        const filter = {};
        if (params.userId)
            filter.userId = new mongoose_1.default.Types.ObjectId(params.userId);
        if (params.walletId)
            filter.walletId = new mongoose_1.default.Types.ObjectId(params.walletId);
        if (params.type)
            filter.type = params.type;
        if (params.category)
            filter.category = params.category;
        if (params.status)
            filter.status = params.status;
        if (params.currency)
            filter.currency = params.currency;
        if (params.dateFrom || params.dateTo) {
            filter.createdAt = {};
            if (params.dateFrom)
                filter.createdAt.$gte = params.dateFrom;
            if (params.dateTo)
                filter.createdAt.$lte = params.dateTo;
        }
        const stats = await this.model.aggregate([
            { $match: filter },
            {
                $group: {
                    _id: null,
                    totalTransactions: { $sum: 1 },
                    totalAmount: { $sum: '$amount' },
                    totalFees: { $sum: { $ifNull: ['$feeAmount', 0] } },
                    totalVAT: { $sum: { $ifNull: ['$vatAmount', 0] } },
                    avgAmount: { $avg: '$amount' },
                    minAmount: { $min: '$amount' },
                    maxAmount: { $max: '$amount' }
                }
            }
        ]);
        return stats[0] || {
            totalTransactions: 0,
            totalAmount: 0,
            totalFees: 0,
            totalVAT: 0,
            avgAmount: 0,
            minAmount: 0,
            maxAmount: 0
        };
    }
    async findOldTransactions(cutoffDate, batchSize = 1000) {
        return this.model.find({
            createdAt: { $lt: cutoffDate },
            status: { $in: ['completed', 'failed', 'cancelled'] }
        }).limit(batchSize).lean();
    }
    async deleteOldTransactions(cutoffDate) {
        return this.model.deleteMany({
            createdAt: { $lt: cutoffDate },
            status: { $in: ['completed', 'failed', 'cancelled'] }
        });
    }
    async getCategoryBreakdown(startDate, endDate) {
        return this.model.aggregate([
            {
                $match: {
                    createdAt: { $gte: startDate, $lte: endDate }
                }
            },
            {
                $group: {
                    _id: '$category',
                    count: { $sum: 1 },
                    totalAmount: { $sum: '$amount' }
                }
            },
            {
                $sort: { totalAmount: -1 }
            }
        ]);
    }
    async searchTransactions(params, page = 1, limit = 20) {
        const filter = {};
        if (params.userId)
            filter.userId = new mongoose_1.default.Types.ObjectId(params.userId);
        if (params.walletId)
            filter.walletId = new mongoose_1.default.Types.ObjectId(params.walletId);
        if (params.type)
            filter.type = params.type;
        if (params.category)
            filter.category = params.category;
        if (params.status)
            filter.status = params.status;
        if (params.currency)
            filter.currency = params.currency;
        if (params.reference)
            filter.reference = { $regex: params.reference, $options: 'i' };
        if (params.externalReference)
            filter.externalReference = { $regex: params.externalReference, $options: 'i' };
        if (params.journalEntryId)
            filter.journalEntryId = params.journalEntryId;
        if (params.isReversal !== undefined)
            filter.isReversal = params.isReversal;
        if (params.minAmount !== undefined)
            filter.amount = { ...filter.amount, $gte: params.minAmount };
        if (params.maxAmount !== undefined)
            filter.amount = { ...filter.amount, $lte: params.maxAmount };
        if (params.dateFrom || params.dateTo) {
            filter.createdAt = {};
            if (params.dateFrom)
                filter.createdAt.$gte = params.dateFrom;
            if (params.dateTo)
                filter.createdAt.$lte = params.dateTo;
        }
        const skip = (page - 1) * limit;
        const [data, total] = await Promise.all([
            this.model.find(filter).sort({ createdAt: -1 }).skip(skip).limit(limit).lean(),
            this.model.countDocuments(filter)
        ]);
        return {
            data,
            pagination: {
                page,
                limit,
                total,
                pages: Math.ceil(total / limit)
            }
        };
    }
}
exports.TransactionRepository = TransactionRepository;
exports.transactionRepository = new TransactionRepository();
