"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.serviceProviderRepository = exports.ServiceProviderRepository = void 0;
const service_provider_1 = require("../models/service-provider");
const base_1 = require("./base");
class ServiceProviderRepository extends base_1.BaseRepository {
    model = service_provider_1.ServiceProvider;
    constructor() {
        super();
    }
    async findByServiceType(serviceType, filters = {}) {
        const query = {
            serviceType,
            isDeleted: { $ne: true },
            ...filters,
        };
        return this.model.find(query)
            .sort({ priority: 1, createdAt: -1 })
            .exec();
    }
    async findBySlug(slug) {
        return this.model.findOne({
            slug,
            isDeleted: { $ne: true }
        }).exec();
    }
    async findActiveProviders(serviceType) {
        const query = {
            status: 'active',
            isHealthy: true,
            isDeleted: { $ne: true },
        };
        if (serviceType) {
            query.serviceType = serviceType;
        }
        return this.model.find(query)
            .sort({ priority: 1, successRate: -1 })
            .exec();
    }
    async findByPriority(serviceType, limit = 5) {
        return this.model.find({
            serviceType,
            status: 'active',
            isHealthy: true,
            isDeleted: { $ne: true },
        })
            .sort({ priority: 1, successRate: -1 })
            .limit(limit)
            .exec();
    }
    async updateHealthStatus(id, healthData) {
        return this.model.findByIdAndUpdate(id, {
            ...healthData,
            lastHealthCheck: healthData.lastHealthCheck || new Date(),
        }, { new: true }).exec();
    }
    async incrementErrorCount(id) {
        return this.model.findByIdAndUpdate(id, {
            $inc: { errorCount: 1 },
            $set: { lastHealthCheck: new Date() }
        }, { new: true }).exec();
    }
    async resetErrorCount(id) {
        return this.model.findByIdAndUpdate(id, {
            errorCount: 0,
            isHealthy: true,
            lastHealthCheck: new Date()
        }, { new: true }).exec();
    }
    async findBySupportedService(service, serviceType) {
        const query = {
            supportedServices: service,
            status: 'active',
            isHealthy: true,
            isDeleted: { $ne: true },
        };
        if (serviceType) {
            query.serviceType = serviceType;
        }
        return this.model.find(query)
            .sort({ priority: 1, successRate: -1 })
            .exec();
    }
    async getProviderStats() {
        const [stats] = await this.model.aggregate([
            {
                $match: { isDeleted: { $ne: true } }
            },
            {
                $group: {
                    _id: null,
                    total: { $sum: 1 },
                    active: { $sum: { $cond: [{ $eq: ['$status', 'active'] }, 1, 0] } },
                    inactive: { $sum: { $cond: [{ $ne: ['$status', 'active'] }, 1, 0] } },
                    healthy: { $sum: { $cond: ['$isHealthy', 1, 0] } },
                }
            }
        ]);
        const serviceTypeStats = await this.model.aggregate([
            {
                $match: { isDeleted: { $ne: true } }
            },
            {
                $group: {
                    _id: '$serviceType',
                    count: { $sum: 1 }
                }
            }
        ]);
        const byServiceType = serviceTypeStats.reduce((acc, item) => {
            acc[item._id] = item.count;
            return acc;
        }, {});
        return {
            total: stats?.total || 0,
            active: stats?.active || 0,
            inactive: stats?.inactive || 0,
            healthy: stats?.healthy || 0,
            byServiceType,
        };
    }
    async searchProviders(searchTerm, filters = {}) {
        const query = {
            $and: [
                { isDeleted: { $ne: true } },
                {
                    $or: [
                        { name: { $regex: searchTerm, $options: 'i' } },
                        { displayName: { $regex: searchTerm, $options: 'i' } },
                        { slug: { $regex: searchTerm, $options: 'i' } },
                        { supportedServices: { $in: [new RegExp(searchTerm, 'i')] } },
                    ]
                }
            ]
        };
        Object.entries(filters).forEach(([key, value]) => {
            if (value !== undefined) {
                query[key] = value;
            }
        });
        return this.model.find(query)
            .sort({ priority: 1, name: 1 })
            .exec();
    }
}
exports.ServiceProviderRepository = ServiceProviderRepository;
exports.serviceProviderRepository = new ServiceProviderRepository();
