"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.servicePlanRepository = exports.ServicePlanRepository = void 0;
const service_plan_1 = require("../models/service-plan");
const base_1 = require("./base");
const mongoose_1 = require("mongoose");
class ServicePlanRepository extends base_1.BaseRepository {
    model = service_plan_1.ServicePlan;
    constructor() {
        super();
    }
    async findByType(type, filters = {}) {
        const query = {
            type,
            isDeleted: { $ne: true },
            ...filters,
        };
        return this.model.find(query)
            .populate('providerId', 'name displayName status isHealthy')
            .sort({ sellingPrice: 1, name: 1 })
            .exec();
    }
    async findByNetwork(network, type) {
        const query = {
            network,
            status: 'active',
            isVisible: true,
            isDeleted: { $ne: true },
        };
        if (type) {
            query.type = type;
        }
        return this.model.find(query)
            .populate('providerId', 'name displayName status isHealthy priority')
            .sort({ sellingPrice: 1, name: 1 })
            .exec();
    }
    async findByProvider(providerId, filters = {}) {
        const query = {
            providerId: new mongoose_1.Types.ObjectId(providerId),
            isDeleted: { $ne: true },
            ...filters,
        };
        return this.model.find(query)
            .sort({ type: 1, sellingPrice: 1 })
            .exec();
    }
    async findActivePlans(filters = {}) {
        const query = {
            status: 'active',
            isVisible: true,
            isDeleted: { $ne: true },
        };
        Object.entries(filters).forEach(([key, value]) => {
            if (value !== undefined) {
                if (key === 'providerId') {
                    query[key] = new mongoose_1.Types.ObjectId(value);
                }
                else if (key === 'minPrice') {
                    query.sellingPrice = { ...query.sellingPrice, $gte: value };
                }
                else if (key === 'maxPrice') {
                    query.sellingPrice = { ...query.sellingPrice, $lte: value };
                }
                else {
                    query[key] = value;
                }
            }
        });
        return this.model.find(query)
            .populate('providerId', 'name displayName status isHealthy priority')
            .sort({ sellingPrice: 1, name: 1 })
            .exec();
    }
    async findByPlanId(planId, providerId) {
        const query = {
            planId,
            isDeleted: { $ne: true },
        };
        if (providerId) {
            query.providerId = new mongoose_1.Types.ObjectId(providerId);
        }
        return this.model.findOne(query)
            .populate('providerId')
            .exec();
    }
    async findDataPlans(network, dataPlanType) {
        const query = {
            type: 'data',
            network,
            status: 'active',
            isVisible: true,
            isDeleted: { $ne: true },
        };
        if (dataPlanType) {
            query.dataPlanType = dataPlanType;
        }
        return this.model.find(query)
            .populate('providerId', 'name displayName status isHealthy priority')
            .sort({ sellingPrice: 1, size: 1 })
            .exec();
    }
    async findElectricityPlans(discoName) {
        const query = {
            type: 'electricity',
            status: 'active',
            isVisible: true,
            isDeleted: { $ne: true },
        };
        if (discoName) {
            query['metadata.discoName'] = new RegExp(discoName, 'i');
        }
        return this.model.find(query)
            .populate('providerId', 'name displayName status isHealthy priority')
            .sort({ 'metadata.discoName': 1, sellingPrice: 1 })
            .exec();
    }
    async findCablePlans(cableProvider) {
        const query = {
            type: 'cable',
            status: 'active',
            isVisible: true,
            isDeleted: { $ne: true },
        };
        if (cableProvider) {
            query['metadata.cableName'] = new RegExp(cableProvider, 'i');
        }
        return this.model.find(query)
            .populate('providerId', 'name displayName status isHealthy priority')
            .sort({ 'metadata.cableName': 1, sellingPrice: 1 })
            .exec();
    }
    async updateSalesStats(id, saleAmount, success) {
        const updateData = {
            $inc: { totalSales: success ? 1 : 0 },
            $set: { lastSaleDate: new Date() }
        };
        if (success) {
            updateData.$min = { successRate: 100 };
        }
        else {
            updateData.$inc.errorCount = 1;
        }
        return this.model.findByIdAndUpdate(id, updateData, { new: true }).exec();
    }
    async searchPlans(searchTerm, filters = {}) {
        const query = {
            $and: [
                { isDeleted: { $ne: true } },
                {
                    $or: [
                        { name: { $regex: searchTerm, $options: 'i' } },
                        { description: { $regex: searchTerm, $options: 'i' } },
                        { planId: { $regex: searchTerm, $options: 'i' } },
                        { 'metadata.discoName': { $regex: searchTerm, $options: 'i' } },
                        { 'metadata.cableName': { $regex: searchTerm, $options: 'i' } },
                    ]
                }
            ]
        };
        Object.entries(filters).forEach(([key, value]) => {
            if (value !== undefined) {
                if (key === 'providerId') {
                    query[key] = new mongoose_1.Types.ObjectId(value);
                }
                else if (key === 'minPrice') {
                    query.sellingPrice = { ...query.sellingPrice, $gte: value };
                }
                else if (key === 'maxPrice') {
                    query.sellingPrice = { ...query.sellingPrice, $lte: value };
                }
                else {
                    query[key] = value;
                }
            }
        });
        return this.model.find(query)
            .populate('providerId', 'name displayName status isHealthy')
            .sort({ sellingPrice: 1, name: 1 })
            .exec();
    }
    async getPlanStats() {
        const [stats] = await this.model.aggregate([
            {
                $match: { isDeleted: { $ne: true } }
            },
            {
                $group: {
                    _id: null,
                    total: { $sum: 1 },
                    active: { $sum: { $cond: [{ $eq: ['$status', 'active'] }, 1, 0] } },
                    inactive: { $sum: { $cond: [{ $ne: ['$status', 'active'] }, 1, 0] } },
                    visible: { $sum: { $cond: ['$isVisible', 1, 0] } },
                    averagePrice: { $avg: '$sellingPrice' },
                }
            }
        ]);
        const typeStats = await this.model.aggregate([
            { $match: { isDeleted: { $ne: true } } },
            { $group: { _id: '$type', count: { $sum: 1 } } }
        ]);
        const networkStats = await this.model.aggregate([
            { $match: { isDeleted: { $ne: true }, network: { $exists: true } } },
            { $group: { _id: '$network', count: { $sum: 1 } } }
        ]);
        const byType = typeStats.reduce((acc, item) => {
            acc[item._id] = item.count;
            return acc;
        }, {});
        const byNetwork = networkStats.reduce((acc, item) => {
            acc[item._id] = item.count;
            return acc;
        }, {});
        return {
            total: stats?.total || 0,
            active: stats?.active || 0,
            inactive: stats?.inactive || 0,
            visible: stats?.visible || 0,
            byType,
            byNetwork,
            averagePrice: stats?.averagePrice || 0,
        };
    }
    async bulkUpdatePrices(providerId, markupPercentage) {
        const result = await this.model.updateMany({
            providerId: new mongoose_1.Types.ObjectId(providerId),
            isDeleted: { $ne: true }
        }, [
            {
                $set: {
                    sellingPrice: {
                        $multiply: ['$amount', { $add: [1, { $divide: [markupPercentage, 100] }] }]
                    }
                }
            }
        ]);
        return { modifiedCount: result.modifiedCount };
    }
    async findDistinctDataPlanTypes() {
        const distinctTypes = await this.model.distinct('dataPlanType', {
            type: 'data',
            status: 'active',
            isVisible: true,
            isDeleted: { $ne: true },
            dataPlanType: { $exists: true, $ne: null }
        });
        return distinctTypes.filter(type => type && type.trim() !== '');
    }
}
exports.ServicePlanRepository = ServicePlanRepository;
exports.servicePlanRepository = new ServicePlanRepository();
