"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.disputeRepository = exports.DisputeRepository = void 0;
const base_1 = require("./base");
const dispute_1 = require("../models/dispute");
class DisputeRepository extends base_1.BaseRepository {
    model = dispute_1.Dispute;
    isValidId(id) {
        return /^[0-9a-fA-F]{24}$/.test(id);
    }
    async findByTransactionId(transactionId) {
        return this.model.find({ transactionId }).sort({ createdAt: -1 });
    }
    async findActiveDisputes() {
        return this.model.find({
            status: { $nin: ['closed', 'resolved'] }
        }).sort({ priority: 1, createdAt: 1 });
    }
    async findUrgentDisputes() {
        return this.model.find({
            priority: 'urgent',
            status: { $nin: ['closed', 'resolved'] }
        }).sort({ createdAt: 1 });
    }
    async findByType(type) {
        return this.model.find({ type }).sort({ createdAt: -1 });
    }
    async findByStatus(status) {
        return this.model.find({ status }).sort({ createdAt: -1 });
    }
    async update(disputeId, updateData) {
        if (!this.isValidId(disputeId))
            return null;
        return await this.model.findByIdAndUpdate(disputeId, updateData, { new: true });
    }
    async getDisputeStatsAggregation() {
        return this.model.aggregate([
            {
                $group: {
                    _id: null,
                    totalDisputes: { $sum: 1 },
                    openDisputes: { $sum: { $cond: [{ $eq: ['$status', 'opened'] }, 1, 0] } },
                    underReview: { $sum: { $cond: [{ $eq: ['$status', 'under_review'] }, 1, 0] } },
                    resolved: { $sum: { $cond: [{ $eq: ['$status', 'resolved'] }, 1, 0] } },
                    closed: { $sum: { $cond: [{ $eq: ['$status', 'closed'] }, 1, 0] } },
                    urgentPriority: { $sum: { $cond: [{ $eq: ['$priority', 'urgent'] }, 1, 0] } },
                    chargebacks: { $sum: { $cond: [{ $eq: ['$type', 'chargeback'] }, 1, 0] } }
                }
            }
        ]);
    }
    async getDisputeStats() {
        return this.model.aggregate([
            {
                $group: {
                    _id: null,
                    totalDisputes: { $sum: 1 },
                    openDisputes: { $sum: { $cond: [{ $eq: ['$status', 'opened'] }, 1, 0] } },
                    underReview: { $sum: { $cond: [{ $eq: ['$status', 'under_review'] }, 1, 0] } },
                    resolved: { $sum: { $cond: [{ $eq: ['$status', 'resolved'] }, 1, 0] } },
                    closed: { $sum: { $cond: [{ $eq: ['$status', 'closed'] }, 1, 0] } },
                    urgentPriority: { $sum: { $cond: [{ $eq: ['$priority', 'urgent'] }, 1, 0] } },
                    chargebacks: { $sum: { $cond: [{ $eq: ['$type', 'chargeback'] }, 1, 0] } }
                }
            }
        ]);
    }
}
exports.DisputeRepository = DisputeRepository;
exports.disputeRepository = new DisputeRepository();
