"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseRepository = void 0;
class BaseRepository {
    softDeleteEnabled = false;
    async create(data) {
        const document = new this.model(data);
        return await document.save();
    }
    async findById(id) {
        if (!this.isValidId(id)) {
            return null;
        }
        return await this.model.findById(id);
    }
    async find(filter = {}) {
        const finalFilter = this.applySoftDeleteFilter(filter);
        return await this.model.find(finalFilter);
    }
    async findOne(filter = {}) {
        const finalFilter = this.applySoftDeleteFilter(filter);
        return await this.model.findOne(finalFilter);
    }
    async updateById(id, data) {
        if (!this.isValidId(id)) {
            return null;
        }
        return await this.model.findByIdAndUpdate(id, data, { new: true, runValidators: true });
    }
    async deleteById(id) {
        if (!this.isValidId(id)) {
            return false;
        }
        if (this.softDeleteEnabled) {
            const result = await this.model.findByIdAndUpdate(id, { $set: { isDeleted: true, deletedAt: new Date() } }, { new: true });
            return result !== null;
        }
        else {
            const result = await this.model.findByIdAndDelete(id);
            return result !== null;
        }
    }
    async softDelete(id) {
        if (!this.isValidId(id)) {
            return null;
        }
        return await this.model.findByIdAndUpdate(id, { isDeleted: true, deletedAt: new Date() }, { new: true });
    }
    async count(filter = {}) {
        const finalFilter = this.applySoftDeleteFilter(filter);
        return await this.model.countDocuments(finalFilter);
    }
    async exists(filter) {
        const count = await this.model.countDocuments(filter);
        return count > 0;
    }
    async findPaginated(filter = {}, page = 1, limit = 10) {
        const skip = (page - 1) * limit;
        const finalFilter = this.applySoftDeleteFilter(filter);
        const [data, total] = await Promise.all([
            this.model.find(finalFilter).skip(skip).limit(limit),
            this.model.countDocuments(finalFilter)
        ]);
        const totalPages = Math.ceil(total / limit);
        const hasNext = page < totalPages;
        const hasPrev = page > 1;
        return {
            data,
            pagination: {
                page,
                limit,
                total,
                totalPages,
                hasNext,
                hasPrev
            }
        };
    }
    isValidId(_id) {
        return true;
    }
    applySoftDeleteFilter(filter) {
        if (!this.softDeleteEnabled)
            return filter;
        return { ...filter, isDeleted: { $ne: true } };
    }
    getModel() {
        return this.model;
    }
}
exports.BaseRepository = BaseRepository;
