"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.setupConfigurableServices = setupConfigurableServices;
exports.up = setupConfigurableServices;
exports.down = down;
const service_provider_1 = require("../models/service-provider");
const service_configuration_1 = require("../models/service-configuration");
const env_1 = require("../../config/env");
async function setupConfigurableServices() {
    console.log('🚀 Setting up configurable services system...');
    try {
        await createDefaultProviders();
        await createSystemConfigurations();
        console.log('✅ Configurable services system setup completed successfully!');
    }
    catch (error) {
        console.error('❌ Failed to setup configurable services system:', error);
        throw error;
    }
}
async function createDefaultProviders() {
    console.log('📡 Creating default service providers...');
    const providers = [
        {
            name: 'ASB Data',
            slug: 'asbdata',
            displayName: 'ASB Data',
            serviceType: 'vtu',
            status: 'active',
            baseUrl: env_1.env.ASB_DATA_BASE_URL || 'https://asbdata.com/api',
            apiKey: env_1.env.ASB_DATA_API_KEY || 'your-asbdata-api-key',
            secretKey: env_1.env.ASB_DATA_SECRET_KEY || '',
            timeout: 30000,
            retryAttempts: 3,
            sandboxMode: env_1.env.NODE_ENV !== 'production',
            priority: 1,
            defaultMarkup: 2.5,
            supportedServices: [
                'airtime',
                'data',
                'electricity',
                'cable',
                'education',
            ],
            supportedNetworks: ['mtn', 'airtel', 'glo', '9mobile'],
            rateLimitPerMinute: 60,
            rateLimitPerHour: 1000,
            isHealthy: true,
            errorCount: 0,
            successRate: 100,
        },
        {
            name: 'Mock Provider',
            slug: 'mock',
            displayName: 'Mock Provider (Testing)',
            serviceType: 'vtu',
            status: env_1.env.NODE_ENV === 'production' ? 'inactive' : 'active',
            baseUrl: 'https://mock.api',
            apiKey: 'mock-api-key',
            timeout: 5000,
            retryAttempts: 1,
            sandboxMode: true,
            priority: 10,
            defaultMarkup: 0,
            supportedServices: ['airtime', 'data'],
            supportedNetworks: ['mtn', 'airtel', 'glo', '9mobile'],
            isHealthy: true,
            errorCount: 0,
            successRate: 100,
        },
        {
            name: 'Bilal Sada Sub',
            slug: 'bilal-sada-sub',
            displayName: 'Bilal Sada Sub',
            serviceType: 'vtu',
            status: 'inactive',
            baseUrl: 'https://bilalsadasub.com/api',
            apiKey: 'your-bilal-sada-api-key',
            timeout: 30000,
            retryAttempts: 3,
            sandboxMode: false,
            priority: 2,
            defaultMarkup: 3.0,
            supportedServices: ['airtime', 'data', 'electricity', 'cable'],
            supportedNetworks: ['mtn', 'airtel', 'glo', '9mobile'],
            isHealthy: false,
            errorCount: 0,
            successRate: 0,
        },
    ];
    for (const providerData of providers) {
        const existingProvider = await service_provider_1.ServiceProvider.findOne({
            slug: providerData.slug,
        });
        if (!existingProvider) {
            await service_provider_1.ServiceProvider.create(providerData);
            console.log(`✅ Created provider: ${providerData.displayName}`);
        }
        else {
            console.log(`⚠️ Provider already exists: ${providerData.displayName}`);
        }
    }
}
async function createSystemConfigurations() {
    console.log('⚙️ Creating system configurations...');
    const configurations = [
        {
            key: 'default_vtu_provider',
            scope: 'global',
            type: 'string',
            value: 'asbdata',
            name: 'Default VTU Provider',
            description: 'Default provider for VTU services',
            category: 'providers',
            isPublic: false,
            isEditable: true,
        },
        {
            key: 'max_daily_vtu_limit',
            scope: 'global',
            type: 'number',
            value: 50000,
            name: 'Maximum Daily VTU Limit',
            description: 'Maximum amount a user can spend on VTU per day (in Naira)',
            category: 'limits',
            isPublic: true,
            isEditable: true,
            validation: {
                required: true,
                min: 1000,
                max: 1000000,
            },
        },
        {
            key: 'enable_provider_fallback',
            scope: 'global',
            type: 'boolean',
            value: true,
            name: 'Enable Provider Fallback',
            description: 'Automatically fallback to next provider if primary fails',
            category: 'providers',
            isPublic: false,
            isEditable: true,
        },
        {
            key: 'cache_plans_ttl',
            scope: 'global',
            type: 'number',
            value: 300,
            name: 'Plans Cache TTL',
            description: 'Time to live for cached plans in seconds',
            category: 'performance',
            isPublic: false,
            isEditable: true,
            validation: {
                min: 60,
                max: 3600,
            },
        },
        {
            key: 'maintenance_mode',
            scope: 'global',
            type: 'boolean',
            value: false,
            name: 'Maintenance Mode',
            description: 'Enable maintenance mode for all services',
            category: 'system',
            isPublic: true,
            isEditable: true,
        },
        {
            key: 'vat_rate',
            scope: 'service',
            type: 'object',
            serviceType: 'airtime',
            value: { rate: 0, enabled: false },
            defaultValue: { rate: 0, enabled: false },
            name: 'VAT for Airtime',
            description: 'VAT configuration for Airtime transactions',
            category: 'taxes',
            isPublic: false,
            isEditable: true,
        },
        {
            key: 'vat_rate',
            scope: 'service',
            type: 'object',
            serviceType: 'data',
            value: { rate: 0, enabled: false },
            defaultValue: { rate: 0, enabled: false },
            name: 'VAT for Data',
            description: 'VAT configuration for Data transactions',
            category: 'taxes',
            isPublic: false,
            isEditable: true,
        },
        {
            key: 'vat_rate',
            scope: 'service',
            type: 'object',
            serviceType: 'bill_payment',
            value: { rate: 0, enabled: false },
            defaultValue: { rate: 0, enabled: false },
            name: 'VAT for Bills',
            description: 'VAT configuration for Bills transactions',
            category: 'taxes',
            isPublic: false,
            isEditable: true,
        },
        {
            key: 'vat_rate',
            scope: 'service',
            type: 'object',
            serviceType: 'education',
            value: { rate: 0, enabled: false },
            defaultValue: { rate: 0, enabled: false },
            name: 'VAT for Education',
            description: 'VAT configuration for Education transactions',
            category: 'taxes',
            isPublic: false,
            isEditable: true,
        },
    ];
    for (const configData of configurations) {
        const existingConfig = await service_configuration_1.ServiceConfiguration.findOne({
            key: configData.key,
            scope: configData.scope,
        });
        if (!existingConfig) {
            await service_configuration_1.ServiceConfiguration.create(configData);
            console.log(`✅ Created configuration: ${configData.name}`);
        }
        else {
            console.log(`⚠️ Configuration already exists: ${configData.name}`);
        }
    }
}
async function down() {
    console.log('🔄 Rolling back configurable services setup...');
    try {
        await service_configuration_1.ServiceConfiguration.deleteMany({});
        await service_provider_1.ServiceProvider.deleteMany({});
        console.log('✅ Rollback completed successfully!');
    }
    catch (error) {
        console.error('❌ Rollback failed:', error);
        throw error;
    }
}
