"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CONNECTION_STATUS = exports.databaseManager = exports.getDatabaseStats = exports.waitForDatabase = exports.testConnection = exports.performHealthCheck = exports.isDatabaseConnected = exports.getDatabaseStatus = exports.closeDatabase = exports.initializeDatabase = void 0;
const index_1 = require("./index");
Object.defineProperty(exports, "databaseManager", { enumerable: true, get: function () { return index_1.databaseManager; } });
const logger_1 = require("../config/logger");
const initializeDatabase = async () => {
    try {
        logger_1.logger.info('Initializing database connection...');
        await index_1.databaseManager.connect();
        logger_1.logger.info('Database connection initialized successfully');
    }
    catch (error) {
        logger_1.logger.error('Failed to initialize database connection:', error);
        throw error;
    }
};
exports.initializeDatabase = initializeDatabase;
const closeDatabase = async () => {
    try {
        logger_1.logger.info('Closing database connection...');
        await index_1.databaseManager.disconnect();
        logger_1.logger.info('Database connection closed successfully');
    }
    catch (error) {
        logger_1.logger.error('Failed to close database connection:', error);
        throw error;
    }
};
exports.closeDatabase = closeDatabase;
const getDatabaseStatus = () => {
    return index_1.databaseManager.getStatus();
};
exports.getDatabaseStatus = getDatabaseStatus;
const isDatabaseConnected = () => {
    return index_1.databaseManager.isDatabaseConnected();
};
exports.isDatabaseConnected = isDatabaseConnected;
const performHealthCheck = async () => {
    try {
        const isHealthy = await index_1.databaseManager.healthCheck();
        if (isHealthy) {
            logger_1.logger.info('Database health check passed');
        }
        else {
            logger_1.logger.warn('Database health check failed');
        }
        return isHealthy;
    }
    catch (error) {
        logger_1.logger.error('Database health check error:', error);
        return false;
    }
};
exports.performHealthCheck = performHealthCheck;
const testConnection = async () => {
    try {
        logger_1.logger.info('Testing database connection...');
        if (!(0, exports.isDatabaseConnected)()) {
            await (0, exports.initializeDatabase)();
        }
        const isHealthy = await (0, exports.performHealthCheck)();
        if (isHealthy) {
            logger_1.logger.info('Database connection test successful');
            return true;
        }
        else {
            logger_1.logger.error('Database connection test failed');
            return false;
        }
    }
    catch (error) {
        logger_1.logger.error('Database connection test error:', error);
        return false;
    }
};
exports.testConnection = testConnection;
const waitForDatabase = async (maxAttempts = 30, delayMs = 1000) => {
    logger_1.logger.info('Waiting for database to be ready...');
    for (let attempt = 1; attempt <= maxAttempts; attempt++) {
        try {
            if ((0, exports.isDatabaseConnected)()) {
                const isHealthy = await (0, exports.performHealthCheck)();
                if (isHealthy) {
                    logger_1.logger.info(`Database is ready after ${attempt} attempts`);
                    return true;
                }
            }
            if (attempt < maxAttempts) {
                logger_1.logger.info(`Database not ready, attempt ${attempt}/${maxAttempts}, retrying in ${delayMs}ms...`);
                await new Promise(resolve => setTimeout(resolve, delayMs));
            }
        }
        catch (error) {
            logger_1.logger.warn(`Database connection attempt ${attempt} failed:`, error);
            if (attempt < maxAttempts) {
                await new Promise(resolve => setTimeout(resolve, delayMs));
            }
        }
    }
    logger_1.logger.error(`Database failed to become ready after ${maxAttempts} attempts`);
    return false;
};
exports.waitForDatabase = waitForDatabase;
const getDatabaseStats = async () => {
    try {
        if (!(0, exports.isDatabaseConnected)()) {
            throw new Error('Database not connected');
        }
        const connection = index_1.databaseManager.getConnection();
        const db = connection.db;
        if (!db) {
            throw new Error('Database instance not available');
        }
        const stats = await db.stats();
        return {
            collections: stats['collections'],
            dataSize: stats['dataSize'],
            storageSize: stats['storageSize'],
            indexes: stats['indexes'],
            indexSize: stats['indexSize'],
            objects: stats['objects'],
            avgObjSize: stats['avgObjSize'],
            fileSize: stats['fileSize']
        };
    }
    catch (error) {
        logger_1.logger.error('Failed to get database stats:', error);
        throw error;
    }
};
exports.getDatabaseStats = getDatabaseStats;
exports.CONNECTION_STATUS = {
    DISCONNECTED: 'disconnected',
    CONNECTED: 'connected',
    CONNECTING: 'connecting',
    DISCONNECTING: 'disconnecting'
};
