"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.notifier = exports.Notifier = void 0;
const nodemailer_1 = __importDefault(require("nodemailer"));
const env_1 = require("../../config/env");
class Notifier {
    mailer = env_1.env.SMTP_HOST && env_1.env.SMTP_USER && env_1.env.SMTP_PASS ? nodemailer_1.default.createTransport({
        host: env_1.env.SMTP_HOST,
        port: env_1.env.SMTP_PORT,
        secure: env_1.env.SMTP_PORT === 465,
        auth: { user: env_1.env.SMTP_USER, pass: env_1.env.SMTP_PASS },
    }) : null;
    async sendEmail(to, subject, text, html) {
        if (!this.mailer) {
            console.log(`[DEV] Email would be sent to ${to}: ${subject}`);
            console.log(`[DEV] Email content: ${text}`);
            return;
        }
        if (env_1.env.NODE_ENV === 'development' && !env_1.env.SMTP_PASS) {
            console.log(`[DEV] Email would be sent to ${to}: ${subject}`);
            console.log(`[DEV] Email content: ${text}`);
            if (html) {
                console.log(`[DEV] HTML content: ${html}`);
            }
            return;
        }
        try {
            console.log(`[Notifier] Sending email to ${to}: ${subject}`);
            const mailOptions = {
                from: env_1.env.SMTP_USER,
                to,
                subject,
                text,
            };
            if (html) {
                mailOptions.html = html;
            }
            const timeoutPromise = new Promise((_, reject) => {
                setTimeout(() => reject(new Error('Email sending timeout')), 15000);
            });
            const sendPromise = this.mailer.sendMail(mailOptions);
            await Promise.race([sendPromise, timeoutPromise]);
            console.log(`[Notifier] Email sent successfully to ${to}`);
        }
        catch (error) {
            if (error instanceof Error && error.message === 'Email sending timeout') {
                console.warn(`[Notifier] Email sending timed out for ${to}`);
            }
            else {
                console.warn(`[Notifier] Email sending failed for ${to}:`, error);
            }
        }
    }
    async sendSms(phone, text) {
        if (!env_1.env.SMS_API_URL || !env_1.env.SMS_API_KEY) {
            console.log(`[DEV] SMS would be sent to ${phone}: ${text}`);
            return;
        }
        if (env_1.env.NODE_ENV === 'development' && !env_1.env.SMS_API_KEY) {
            console.log(`[DEV] SMS would be sent to ${phone}: ${text}`);
            return;
        }
        try {
            console.log(`[Notifier] Sending SMS to ${phone}: ${text}`);
            const controller = new AbortController();
            const timeoutId = setTimeout(() => controller.abort(), 10000);
            const response = await fetch(env_1.env.SMS_API_URL, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${env_1.env.SMS_API_KEY}`
                },
                body: JSON.stringify({ to: phone, message: text }),
                signal: controller.signal
            });
            clearTimeout(timeoutId);
            if (!response.ok) {
                console.warn(`[Notifier] SMS API returned ${response.status}: ${response.statusText}`);
            }
            else {
                console.log(`[Notifier] SMS sent successfully to ${phone}`);
            }
        }
        catch (error) {
            if (error instanceof Error && error.name === 'AbortError') {
                console.warn(`[Notifier] SMS request timed out for ${phone}`);
            }
            else {
                console.warn(`[Notifier] SMS sending failed for ${phone}:`, error);
            }
        }
    }
}
exports.Notifier = Notifier;
exports.notifier = new Notifier();
