"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.securityMonitoring = exports.securityUtils = exports.createSecurityMiddleware = exports.securityHeaders = exports.hppOptions = exports.enhancedHelmetOptions = exports.getCSPPolicy = exports.strictCorsOptions = void 0;
const helmet_1 = __importDefault(require("helmet"));
const cors_1 = __importDefault(require("cors"));
const hpp_1 = __importDefault(require("hpp"));
const env_1 = require("../../config/env");
exports.strictCorsOptions = {
    origin: (origin, callback) => {
        if (!origin)
            return callback(null, true);
        if (env_1.env.NODE_ENV === 'development') {
            return callback(null, true);
        }
        const allowedOrigins = env_1.env.CORS_ORIGIN.split(',').map(o => o.trim());
        if (allowedOrigins.includes(origin) || allowedOrigins.includes('*')) {
            callback(null, true);
        }
        else {
            callback(new Error('Not allowed by CORS'));
        }
    },
    credentials: env_1.env.CORS_CREDENTIALS,
    methods: ['GET', 'POST', 'PUT', 'DELETE', 'PATCH', 'OPTIONS'],
    allowedHeaders: [
        'Origin',
        'X-Requested-With',
        'Content-Type',
        'Accept',
        'Authorization',
        'X-API-Key',
        'X-Request-ID',
        'X-Correlation-ID',
        'X-Client-Version',
        'X-Platform',
        'X-App-Version',
        'X-Timestamp',
    ],
    exposedHeaders: [
        'X-Total-Count',
        'X-Page-Count',
        'X-Request-ID',
        'X-RateLimit-Limit',
        'X-RateLimit-Remaining',
        'X-RateLimit-Reset'
    ],
    maxAge: 86400,
    preflightContinue: false,
    optionsSuccessStatus: 204,
};
const getCSPPolicy = (environment) => {
    const basePolicy = {
        'default-src': ["'self'"],
        'script-src': [
            "'self'",
            "'unsafe-inline'",
            'https://js.stripe.com',
            'https://checkout.stripe.com',
        ],
        'style-src': [
            "'self'",
            "'unsafe-inline'",
            'https://fonts.googleapis.com',
            'https://cdnjs.cloudflare.com',
        ],
        'img-src': [
            "'self'",
            "data:",
            "https:",
            'https://stripe.com',
            'https://checkout.stripe.com',
        ],
        'connect-src': [
            "'self'",
            'https://api.stripe.com',
            'https://api.paystack.co',
            'https://api.flutterwave.com',
            'https://api.nimc.gov.ng',
            'https://api.bvn.ng',
            'wss://localhost:*',
        ],
        'font-src': [
            "'self'",
            'https://fonts.gstatic.com',
            'https://cdnjs.cloudflare.com',
        ],
        'object-src': ["'none'"],
        'media-src': ["'self'"],
        'frame-src': [
            "'none'",
            'https://js.stripe.com',
            'https://checkout.stripe.com',
        ],
        'frame-ancestors': ["'none'"],
        'form-action': ["'self'"],
        'base-uri': ["'self'"],
        'manifest-src': ["'self'"],
        'worker-src': ["'self'"],
        'child-src': ["'none'"],
        'upgrade-insecure-requests': [],
    };
    if (environment === 'development') {
        basePolicy['script-src'] = [
            "'self'",
            "'unsafe-inline'",
            "'unsafe-eval'",
            'https://js.stripe.com',
        ];
        basePolicy['connect-src'] = [
            "'self'",
            'ws://localhost:*',
            'wss://localhost:*',
            'https://api.stripe.com',
            'https://api.paystack.co',
            'https://api.flutterwave.com',
        ];
    }
    return basePolicy;
};
exports.getCSPPolicy = getCSPPolicy;
exports.enhancedHelmetOptions = {
    contentSecurityPolicy: {
        directives: (0, exports.getCSPPolicy)(env_1.env.NODE_ENV),
        reportOnly: false,
    },
    crossOriginEmbedderPolicy: {
        policy: 'require-corp',
    },
    crossOriginOpenerPolicy: {
        policy: 'same-origin',
    },
    crossOriginResourcePolicy: {
        policy: 'same-site',
    },
    dnsPrefetchControl: {
        allow: false,
    },
    frameguard: {
        action: 'deny',
    },
    hidePoweredBy: true,
    hsts: {
        maxAge: 31536000,
        includeSubDomains: true,
        preload: true,
    },
    ieNoOpen: true,
    noSniff: true,
    permittedCrossDomainPolicies: {
        permittedPolicies: 'none',
    },
    referrerPolicy: {
        policy: 'strict-origin-when-cross-origin',
    },
    xssFilter: true,
};
exports.hppOptions = {
    whitelist: [
        'filter',
        'sort',
        'page',
        'limit',
        'fields',
        'populate',
    ],
    checkBody: true,
    checkBodyOnlyForContentTypes: ['application/json', 'application/x-www-form-urlencoded'],
    checkQuery: true,
    checkQueryOnlyForContentTypes: ['application/json', 'application/x-www-form-urlencoded'],
};
exports.securityHeaders = {
    'X-Content-Type-Options': 'nosniff',
    'X-Frame-Options': 'DENY',
    'X-XSS-Protection': '1; mode=block',
    'Referrer-Policy': 'strict-origin-when-cross-origin',
    'Permissions-Policy': [
        'geolocation=()',
        'microphone=()',
        'camera=()',
        'payment=()',
        'usb=()',
        'magnetometer=()',
        'gyroscope=()',
        'accelerometer=()',
    ].join(', '),
    'Strict-Transport-Security': 'max-age=31536000; includeSubDomains; preload',
    'X-DNS-Prefetch-Control': 'off',
    'X-Download-Options': 'noopen',
    'X-Permitted-Cross-Domain-Policies': 'none',
    'X-Robots-Tag': 'noindex, nofollow',
    'X-Request-ID': '',
    'X-Correlation-ID': '',
};
const createSecurityMiddleware = () => {
    return {
        helmet: (0, helmet_1.default)(exports.enhancedHelmetOptions),
        cors: (0, cors_1.default)(exports.strictCorsOptions),
        hpp: (0, hpp_1.default)(exports.hppOptions),
    };
};
exports.createSecurityMiddleware = createSecurityMiddleware;
exports.securityUtils = {
    validateApiKey: (apiKey) => {
        const apiKeyRegex = /^[A-Za-z0-9]{32,64}$/;
        return apiKeyRegex.test(apiKey);
    },
    validateJWTFormat: (token) => {
        const jwtRegex = /^[A-Za-z0-9-_]+\.[A-Za-z0-9-_]+\.[A-Za-z0-9-_]*$/;
        return jwtRegex.test(token);
    },
    sanitizeInput: (input) => {
        return input
            .replace(/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/gi, '')
            .replace(/javascript:/gi, '')
            .replace(/on\w+\s*=/gi, '')
            .replace(/vbscript:/gi, '')
            .replace(/data:/gi, '')
            .replace(/<iframe\b[^<]*(?:(?!<\/iframe>)<[^<]*)*<\/iframe>/gi, '');
    },
    validateFileType: (mimeType, allowedTypes) => {
        return allowedTypes.includes(mimeType);
    },
    validateFileSize: (size, maxSize) => {
        return size <= maxSize;
    },
    generateSecureRandom: (length = 32) => {
        const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
        let result = '';
        for (let i = 0; i < length; i++) {
            result += chars.charAt(Math.floor(Math.random() * chars.length));
        }
        return result;
    },
    validatePasswordStrength: (password) => {
        const errors = [];
        if (password.length < 8) {
            errors.push('Password must be at least 8 characters long');
        }
        if (!/[A-Z]/.test(password)) {
            errors.push('Password must contain at least one uppercase letter');
        }
        if (!/[a-z]/.test(password)) {
            errors.push('Password must contain at least one lowercase letter');
        }
        if (!/[0-9]/.test(password)) {
            errors.push('Password must contain at least one number');
        }
        if (!/[!@#$%^&*(),.?":{}|<>]/.test(password)) {
            errors.push('Password must contain at least one special character');
        }
        return {
            isValid: errors.length === 0,
            errors,
        };
    },
};
exports.securityMonitoring = {
    trackSecurityEvent: (event, details) => {
        console.warn(`🔒 SECURITY EVENT: ${event}`, {
            timestamp: new Date().toISOString(),
            event,
            details,
            environment: env_1.env.NODE_ENV,
        });
    },
    trackFailedAuth: (ip, userAgent, reason) => {
        exports.securityMonitoring.trackSecurityEvent('FAILED_AUTH', {
            ip,
            userAgent,
            reason,
        });
    },
    trackSuspiciousActivity: (ip, activity, details) => {
        exports.securityMonitoring.trackSecurityEvent('SUSPICIOUS_ACTIVITY', {
            ip,
            activity,
            details,
        });
    },
};
exports.default = {
    strictCorsOptions: exports.strictCorsOptions,
    enhancedHelmetOptions: exports.enhancedHelmetOptions,
    hppOptions: exports.hppOptions,
    securityHeaders: exports.securityHeaders,
    createSecurityMiddleware: exports.createSecurityMiddleware,
    securityUtils: exports.securityUtils,
    securityMonitoring: exports.securityMonitoring,
};
