"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.rotateSecret = exports.createSecret = exports.getSecret = exports.encryptionManager = exports.keyRotationScheduler = exports.secretsManager = exports.KeyRotationScheduler = exports.SecretsManager = exports.EncryptionManager = void 0;
const crypto_1 = __importDefault(require("crypto"));
const redis_1 = require("../../config/redis");
const env_1 = require("../../config/env");
class EncryptionManager {
    static instance;
    masterKey;
    constructor() {
        this.masterKey = env_1.env.JWT_SECRET;
    }
    static getInstance() {
        if (!EncryptionManager.instance) {
            EncryptionManager.instance = new EncryptionManager();
        }
        return EncryptionManager.instance;
    }
    encrypt(value) {
        const iv = crypto_1.default.randomBytes(16);
        const key = crypto_1.default.createHash('sha256').update(this.masterKey).digest();
        const cipher = crypto_1.default.createCipheriv('aes-256-gcm', key, iv);
        let encrypted = cipher.update(value, 'utf8', 'hex');
        encrypted += cipher.final('hex');
        const authTag = cipher.getAuthTag();
        return `${iv.toString('hex')}:${authTag.toString('hex')}:${encrypted}`;
    }
    decrypt(encryptedValue) {
        const parts = encryptedValue.split(':');
        if (parts.length !== 3) {
            throw new Error('Invalid encrypted value format');
        }
        const [, authTagHex, encrypted] = parts;
        const iv = Buffer.from(parts[0], 'hex');
        const authTag = Buffer.from(authTagHex, 'hex');
        const key = crypto_1.default.createHash('sha256').update(this.masterKey).digest();
        const decipher = crypto_1.default.createDecipheriv('aes-256-gcm', key, iv);
        decipher.setAuthTag(authTag);
        let decrypted = decipher.update(encrypted, 'hex', 'utf8');
        decrypted += decipher.final('utf8');
        return decrypted;
    }
    generateSecret(length = 32) {
        return crypto_1.default.randomBytes(length).toString('hex');
    }
}
exports.EncryptionManager = EncryptionManager;
class SecretsManager {
    static instance;
    encryptionManager;
    secrets = new Map();
    constructor() {
        this.encryptionManager = EncryptionManager.getInstance();
    }
    static getInstance() {
        if (!SecretsManager.instance) {
            SecretsManager.instance = new SecretsManager();
        }
        return SecretsManager.instance;
    }
    async createSecret(name, value, type, environment = 'development', rotationPeriod = 90) {
        const id = crypto_1.default.randomUUID();
        const encryptedValue = this.encryptionManager.encrypt(value);
        const now = new Date();
        const secret = {
            id,
            name,
            value,
            encryptedValue,
            type,
            environment,
            createdAt: now,
            updatedAt: now,
            rotationPeriod,
            lastRotated: now,
            version: 1,
            isActive: true,
        };
        this.secrets.set(id, secret);
        await this.storeSecretInRedis(secret);
        return secret;
    }
    async getSecret(name, environment) {
        for (const secret of this.secrets.values()) {
            if (secret.name === name && (!environment || secret.environment === environment)) {
                return secret;
            }
        }
        const redisSecret = await this.getSecretFromRedis(name, environment);
        if (redisSecret) {
            this.secrets.set(redisSecret.id, redisSecret);
            return redisSecret;
        }
        return null;
    }
    async getSecretValue(name, environment) {
        const secret = await this.getSecret(name, environment);
        if (!secret)
            return null;
        if (secret.expiresAt && secret.expiresAt < new Date()) {
            throw new Error(`Secret ${name} has expired`);
        }
        return secret.value;
    }
    async rotateSecret(name, environment) {
        const secret = await this.getSecret(name, environment);
        if (!secret) {
            throw new Error(`Secret ${name} not found`);
        }
        const newValue = this.encryptionManager.generateSecret();
        const newEncryptedValue = this.encryptionManager.encrypt(newValue);
        secret.value = newValue;
        secret.encryptedValue = newEncryptedValue;
        secret.version += 1;
        secret.lastRotated = new Date();
        secret.updatedAt = new Date();
        this.secrets.set(secret.id, secret);
        await this.storeSecretInRedis(secret);
        console.log(`🔄 Secret ${name} rotated successfully (version ${secret.version})`);
        return secret;
    }
    async checkRotationNeeded(name, environment) {
        const secret = await this.getSecret(name, environment);
        if (!secret)
            return false;
        const daysSinceRotation = (Date.now() - secret.lastRotated.getTime()) / (1000 * 60 * 60 * 24);
        return daysSinceRotation >= secret.rotationPeriod;
    }
    async rotateExpiredSecrets() {
        const rotatedSecrets = [];
        for (const secret of this.secrets.values()) {
            if (await this.checkRotationNeeded(secret.name, secret.environment)) {
                await this.rotateSecret(secret.name, secret.environment);
                rotatedSecrets.push(secret.name);
            }
        }
        return rotatedSecrets;
    }
    async deactivateSecret(name, environment) {
        const secret = await this.getSecret(name, environment);
        if (!secret) {
            throw new Error(`Secret ${name} not found`);
        }
        secret.isActive = false;
        secret.updatedAt = new Date();
        this.secrets.set(secret.id, secret);
        await this.storeSecretInRedis(secret);
    }
    async listSecrets(environment) {
        const secrets = [];
        for (const secret of this.secrets.values()) {
            if (!environment || secret.environment === environment) {
                secrets.push(secret);
            }
        }
        return secrets;
    }
    async storeSecretInRedis(secret) {
        if (env_1.env.REDIS_OPTIONAL && env_1.env.NODE_ENV !== 'production') {
            return;
        }
        try {
            const client = await redis_1.redisManager.getClient('main');
            const key = `secret:${secret.environment}:${secret.name}`;
            await client.setEx(key, 86400 * 30, JSON.stringify(secret));
        }
        catch (error) {
            if (env_1.env.REDIS_OPTIONAL && env_1.env.NODE_ENV !== 'production') {
                console.warn('Skipping Redis store (optional):', error.message);
                return;
            }
            console.error('Failed to store secret in Redis:', error);
        }
    }
    async getSecretFromRedis(name, environment) {
        if (env_1.env.REDIS_OPTIONAL && env_1.env.NODE_ENV !== 'production') {
            return null;
        }
        try {
            const client = await redis_1.redisManager.getClient('main');
            const envs = environment ? [environment] : ['development', 'production', 'test'];
            for (const env of envs) {
                const key = `secret:${env}:${name}`;
                const data = await client.get(key);
                if (data) {
                    return JSON.parse(data);
                }
            }
        }
        catch (error) {
            if (env_1.env.REDIS_OPTIONAL && env_1.env.NODE_ENV !== 'production') {
                console.warn('Skipping Redis get (optional):', error.message);
                return null;
            }
            console.error('Failed to get secret from Redis:', error);
        }
        return null;
    }
    async initializeDefaultSecrets() {
        const defaultSecrets = [
            {
                name: 'JWT_SECRET',
                value: env_1.env.JWT_SECRET,
                type: 'jwt_secret',
                rotationPeriod: 90,
            },
            {
                name: 'PAYSTACK_SECRET_KEY',
                value: env_1.env.PAYSTACK_SECRET_KEY || '',
                type: 'api_key',
                rotationPeriod: 180,
            },
            {
                name: 'FLUTTERWAVE_SECRET_KEY',
                value: env_1.env.FLUTTERWAVE_SECRET_KEY || '',
                type: 'api_key',
                rotationPeriod: 180,
            },
        ];
        for (const secretConfig of defaultSecrets) {
            if (secretConfig.value) {
                const existing = await this.getSecret(secretConfig.name);
                if (!existing) {
                    await this.createSecret(secretConfig.name, secretConfig.value, secretConfig.type, env_1.env.NODE_ENV, secretConfig.rotationPeriod);
                }
            }
        }
    }
}
exports.SecretsManager = SecretsManager;
class KeyRotationScheduler {
    static instance;
    secretsManager;
    intervalId;
    constructor() {
        this.secretsManager = SecretsManager.getInstance();
    }
    static getInstance() {
        if (!KeyRotationScheduler.instance) {
            KeyRotationScheduler.instance = new KeyRotationScheduler();
        }
        return KeyRotationScheduler.instance;
    }
    startRotationSchedule(checkIntervalHours = 24) {
        if (this.intervalId) {
            clearInterval(this.intervalId);
        }
        this.intervalId = setInterval(async () => {
            try {
                const rotatedSecrets = await this.secretsManager.rotateExpiredSecrets();
                if (rotatedSecrets.length > 0) {
                    console.log(`🔄 Auto-rotated ${rotatedSecrets.length} secrets:`, rotatedSecrets);
                }
            }
            catch (error) {
                console.error('❌ Key rotation failed:', error);
            }
        }, checkIntervalHours * 60 * 60 * 1000);
        console.log(`🔄 Key rotation scheduler started (checking every ${checkIntervalHours} hours)`);
    }
    stopRotationSchedule() {
        if (this.intervalId) {
            clearInterval(this.intervalId);
            this.intervalId = undefined;
            console.log('🔄 Key rotation scheduler stopped');
        }
    }
    async checkAndRotate() {
        return await this.secretsManager.rotateExpiredSecrets();
    }
}
exports.KeyRotationScheduler = KeyRotationScheduler;
exports.secretsManager = SecretsManager.getInstance();
exports.keyRotationScheduler = KeyRotationScheduler.getInstance();
exports.encryptionManager = EncryptionManager.getInstance();
const getSecret = (name, environment) => exports.secretsManager.getSecretValue(name, environment);
exports.getSecret = getSecret;
const createSecret = (name, value, type, environment, rotationPeriod) => exports.secretsManager.createSecret(name, value, type, environment, rotationPeriod);
exports.createSecret = createSecret;
const rotateSecret = (name, environment) => exports.secretsManager.rotateSecret(name, environment);
exports.rotateSecret = rotateSecret;
