"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.FieldEncryption = void 0;
exports.encryptedFieldsPlugin = encryptedFieldsPlugin;
const crypto_1 = __importDefault(require("crypto"));
const env_1 = require("../../config/env");
class FieldEncryption {
    static ALGORITHM = 'aes-256-gcm';
    static IV_LENGTH = 16;
    static TAG_LENGTH = 16;
    static SALT_LENGTH = 32;
    static encryptionKey = null;
    static getEncryptionKey() {
        if (!this.encryptionKey) {
            const masterKey = env_1.env.FIELD_ENCRYPTION_KEY || env_1.env.JWT_SECRET || 'default-encryption-key';
            this.encryptionKey = crypto_1.default.pbkdf2Sync(masterKey, 'unify-field-encryption-salt', 100000, 32, 'sha256');
        }
        return this.encryptionKey;
    }
    static encrypt(plaintext) {
        if (!plaintext)
            return plaintext;
        try {
            const key = this.getEncryptionKey();
            const iv = crypto_1.default.randomBytes(this.IV_LENGTH);
            const cipher = crypto_1.default.createCipheriv(this.ALGORITHM, key, iv);
            cipher.setAAD(Buffer.from('unify-field-encryption'));
            let encrypted = cipher.update(plaintext, 'utf8', 'hex');
            encrypted += cipher.final('hex');
            const tag = cipher.getAuthTag();
            const combined = Buffer.concat([
                iv,
                Buffer.from(encrypted, 'hex'),
                tag
            ]);
            return 'enc:' + combined.toString('base64');
        }
        catch (error) {
            console.error('Field encryption failed:', error);
            throw new Error('Failed to encrypt field');
        }
    }
    static decrypt(encryptedValue) {
        if (!encryptedValue || !encryptedValue.startsWith('enc:')) {
            return encryptedValue;
        }
        try {
            const key = this.getEncryptionKey();
            const combined = Buffer.from(encryptedValue.substring(4), 'base64');
            const iv = combined.subarray(0, this.IV_LENGTH);
            const tag = combined.subarray(-this.TAG_LENGTH);
            const encrypted = combined.subarray(this.IV_LENGTH, -this.TAG_LENGTH);
            const decipher = crypto_1.default.createDecipheriv(this.ALGORITHM, key, iv);
            decipher.setAAD(Buffer.from('unify-field-encryption'));
            decipher.setAuthTag(tag);
            let decrypted = decipher.update(encrypted, undefined, 'utf8');
            decrypted += decipher.final('utf8');
            return decrypted;
        }
        catch (error) {
            console.error('Field decryption failed:', error);
            throw new Error('Failed to decrypt field');
        }
    }
    static isEncrypted(value) {
        return typeof value === 'string' && value.startsWith('enc:');
    }
    static encryptFields(obj, fieldsToEncrypt) {
        const result = { ...obj };
        for (const field of fieldsToEncrypt) {
            if (result[field] && typeof result[field] === 'string') {
                result[field] = this.encrypt(result[field]);
            }
        }
        return result;
    }
    static decryptFields(obj, fieldsToDecrypt) {
        const result = { ...obj };
        for (const field of fieldsToDecrypt) {
            if (result[field] && typeof result[field] === 'string') {
                result[field] = this.decrypt(result[field]);
            }
        }
        return result;
    }
    static createSearchHash(value) {
        if (!value)
            return '';
        const key = this.getEncryptionKey();
        const hmac = crypto_1.default.createHmac('sha256', key);
        hmac.update(value);
        return 'hash:' + hmac.digest('hex');
    }
    static verifyEncryptedField(plainValue, encryptedValue) {
        try {
            const decrypted = this.decrypt(encryptedValue);
            return decrypted === plainValue;
        }
        catch {
            return false;
        }
    }
}
exports.FieldEncryption = FieldEncryption;
function encryptedFieldsPlugin(schema, options) {
    const { fields } = options;
    schema.pre('save', function (next) {
        for (const field of fields) {
            if (this.isModified(field) && this[field] && !FieldEncryption.isEncrypted(this[field])) {
                this[field] = FieldEncryption.encrypt(this[field]);
            }
        }
        next();
    });
    schema.pre(['updateOne', 'findOneAndUpdate', 'updateMany'], function (next) {
        const update = this.getUpdate();
        for (const field of fields) {
            if (update[field] && !FieldEncryption.isEncrypted(update[field])) {
                update[field] = FieldEncryption.encrypt(update[field]);
            }
            if (update.$set && update.$set[field] && !FieldEncryption.isEncrypted(update.$set[field])) {
                update.$set[field] = FieldEncryption.encrypt(update.$set[field]);
            }
        }
        next();
    });
    schema.methods.decryptFields = function (fieldsToDecrypt) {
        const targetFields = fieldsToDecrypt || fields;
        const decrypted = {};
        for (const field of targetFields) {
            if (this[field]) {
                decrypted[field] = FieldEncryption.decrypt(this[field]);
            }
        }
        return decrypted;
    };
    schema.statics.findByEncryptedField = function (field, value) {
        const searchHash = FieldEncryption.createSearchHash(value);
        return this.find({ [`${field}_hash`]: searchHash });
    };
}
exports.default = FieldEncryption;
