"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getPerformanceMetrics = exports.sizeLimit = exports.timeout = exports.sanitizeRequest = exports.securityHeaders = exports.validateRequest = exports.logResponse = exports.logRequest = exports.initializeContext = exports.requestLifecycle = exports.RequestLifecycleMiddleware = void 0;
const perf_hooks_1 = require("perf_hooks");
class RequestLifecycleMiddleware {
    static instance;
    constructor() { }
    static getInstance() {
        if (!RequestLifecycleMiddleware.instance) {
            RequestLifecycleMiddleware.instance = new RequestLifecycleMiddleware();
        }
        return RequestLifecycleMiddleware.instance;
    }
    initializeContext = (req, res, next) => {
        const startTime = perf_hooks_1.performance.now();
        const requestId = req.headers['x-request-id'] ||
            `req_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
        req.context = {
            requestId,
            startTime,
            method: req.method,
            url: req.originalUrl,
            userAgent: req.get('User-Agent') || '',
            ip: req.ip || req.connection.remoteAddress || 'unknown',
            correlationId: req.headers['x-correlation-id'],
        };
        res.setHeader('x-request-id', requestId);
        if (req.context?.correlationId) {
            res.setHeader('x-correlation-id', req.context.correlationId);
        }
        next();
    };
    logRequest = (req, _res, next) => {
        if (!req.context) {
            return next();
        }
        const { method, url, userAgent, ip } = req.context;
        console.log(`📥 ${method} ${url}`, {
            requestId: req.context.requestId,
            userAgent: userAgent?.substring(0, 100),
            ip,
            timestamp: new Date().toISOString(),
        });
        next();
    };
    logResponse = (req, res, next) => {
        if (!req.context) {
            return next();
        }
        const originalSend = res.send;
        const { startTime, method, url } = req.context;
        res.send = function (body) {
            const endTime = perf_hooks_1.performance.now();
            const duration = endTime - startTime;
            const statusCode = res.statusCode;
            const emoji = statusCode >= 400 ? '❌' : '✅';
            console.log(`${emoji} ${method} ${url} - ${statusCode} (${duration.toFixed(2)}ms)`, {
                requestId: req.context?.requestId,
                statusCode,
                duration: `${duration.toFixed(2)}ms`,
                contentLength: body?.length || 0,
                timestamp: new Date().toISOString(),
            });
            return originalSend.call(res, body);
        }.bind(this);
        next();
    };
    performanceMetrics = new Map();
    getPerformanceMetrics() {
        const metrics = {};
        for (const [key, value] of this.performanceMetrics.entries()) {
            metrics[key] = {
                ...value,
                avgDuration: value.count > 0 ? value.totalDuration / value.count : 0,
                errorRate: value.count > 0 ? (value.errorCount / value.count) * 100 : 0,
            };
        }
        return metrics;
    }
    validateRequest = (schema) => {
        return (req, res, next) => {
            if (!schema) {
                return next();
            }
            try {
                const validationResult = this.validateAgainstSchema(req.body, schema);
                if (!validationResult.isValid) {
                    return res.status(400).json({
                        success: false,
                        error: {
                            code: 'VALIDATION_ERROR',
                            message: 'Request validation failed',
                            details: validationResult.errors,
                        },
                        requestId: req.context?.requestId,
                        timestamp: new Date().toISOString(),
                    });
                }
                next();
            }
            catch (error) {
                next(error);
            }
        };
    };
    validateAgainstSchema(_data, _schema) {
        return { isValid: true, errors: [] };
    }
    securityHeaders = (_req, res, next) => {
        res.setHeader('X-Content-Type-Options', 'nosniff');
        res.setHeader('X-Frame-Options', 'DENY');
        res.setHeader('X-XSS-Protection', '1; mode=block');
        res.setHeader('Referrer-Policy', 'strict-origin-when-cross-origin');
        res.setHeader('Permissions-Policy', 'geolocation=(), microphone=(), camera=()');
        next();
    };
    sanitizeRequest = (req, _res, next) => {
        if (req.body) {
            req.body = this.sanitizeData(req.body);
        }
        if (req.query) {
            req.query = this.sanitizeData(req.query);
        }
        next();
    };
    sanitizeData(data) {
        if (typeof data === 'string') {
            return data
                .replace(/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/gi, '')
                .replace(/javascript:/gi, '')
                .replace(/on\w+\s*=/gi, '');
        }
        if (Array.isArray(data)) {
            return data.map(item => this.sanitizeData(item));
        }
        if (typeof data === 'object' && data !== null) {
            const sanitized = {};
            for (const [key, value] of Object.entries(data)) {
                sanitized[key] = this.sanitizeData(value);
            }
            return sanitized;
        }
        return data;
    }
    timeout = (timeoutMs = 30000) => {
        return (req, res, next) => {
            const timeoutId = setTimeout(() => {
                if (!res.headersSent) {
                    res.status(408).json({
                        success: false,
                        error: {
                            code: 'REQUEST_TIMEOUT',
                            message: 'Request timeout',
                        },
                        requestId: req.context?.requestId,
                        timestamp: new Date().toISOString(),
                    });
                }
            }, timeoutMs);
            res.on('finish', () => {
                clearTimeout(timeoutId);
            });
            next();
        };
    };
    sizeLimit = (limit = '10mb') => {
        return (req, res, next) => {
            const contentLength = parseInt(req.headers['content-length'] || '0');
            const limitBytes = this.parseSizeLimit(limit);
            if (contentLength > limitBytes) {
                res.status(413).json({
                    success: false,
                    error: {
                        code: 'PAYLOAD_TOO_LARGE',
                        message: 'Request entity too large',
                    },
                    requestId: req.context?.requestId,
                    timestamp: new Date().toISOString(),
                });
                return;
            }
            next();
        };
    };
    parseSizeLimit(limit) {
        const units = {
            'b': 1,
            'kb': 1024,
            'mb': 1024 * 1024,
            'gb': 1024 * 1024 * 1024,
        };
        const match = limit.match(/^(\d+)([a-z]*)$/i);
        if (!match)
            return 10 * 1024 * 1024;
        const [, size, unit] = match;
        const multiplier = units[unit?.toLowerCase() || 'mb'] || units['mb'];
        return parseInt(size || '0') * (multiplier || 1024 * 1024);
    }
}
exports.RequestLifecycleMiddleware = RequestLifecycleMiddleware;
exports.requestLifecycle = RequestLifecycleMiddleware.getInstance();
exports.initializeContext = exports.requestLifecycle.initializeContext, exports.logRequest = exports.requestLifecycle.logRequest, exports.logResponse = exports.requestLifecycle.logResponse, exports.validateRequest = exports.requestLifecycle.validateRequest, exports.securityHeaders = exports.requestLifecycle.securityHeaders, exports.sanitizeRequest = exports.requestLifecycle.sanitizeRequest, exports.timeout = exports.requestLifecycle.timeout, exports.sizeLimit = exports.requestLifecycle.sizeLimit, exports.getPerformanceMetrics = exports.requestLifecycle.getPerformanceMetrics;
