"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.staticCache = exports.userCache = exports.longCache = exports.mediumCache = exports.shortCache = exports.cacheConfigs = exports.cacheStats = exports.CacheStats = exports.cacheWarmer = exports.CacheWarmer = exports.queryCache = exports.QueryCache = exports.cacheService = exports.cacheInvalidation = exports.cacheMiddleware = exports.CacheKeyGenerator = void 0;
const redis_1 = require("../../config/redis");
const crypto_1 = __importDefault(require("crypto"));
class CacheKeyGenerator {
    static generateKey(req, customKey) {
        if (customKey) {
            return `cache:${customKey}`;
        }
        const { method, originalUrl } = req;
        const query = JSON.stringify(req.query);
        const body = req.method !== 'GET' ? JSON.stringify(req.body) : '';
        const userId = req.user?.userId || 'anonymous';
        const keyData = `${method}:${originalUrl}:${query}:${body}:${userId}`;
        const hash = crypto_1.default.createHash('md5').update(keyData).digest('hex');
        return `cache:${hash}`;
    }
    static generateUserKey(userId, prefix) {
        return `cache:user:${userId}:${prefix}`;
    }
    static generateServiceKey(service, operation, params) {
        const paramHash = crypto_1.default.createHash('md5').update(JSON.stringify(params)).digest('hex');
        return `cache:service:${service}:${operation}:${paramHash}`;
    }
}
exports.CacheKeyGenerator = CacheKeyGenerator;
const cacheMiddleware = (config) => {
    return async (req, res, next) => {
        if (req.method !== 'GET' && !config.condition) {
            return next();
        }
        if (config.condition && !config.condition(req, res)) {
            return next();
        }
        const cacheKey = CacheKeyGenerator.generateKey(req, config.key);
        try {
            const client = await redis_1.redisManager.getClient('cache');
            const cachedData = await client.get(cacheKey);
            if (cachedData) {
                const parsed = JSON.parse(cachedData);
                res.setHeader('X-Cache', 'HIT');
                res.setHeader('X-Cache-Key', cacheKey);
                return res.status(parsed.statusCode || 200).json(parsed.data);
            }
            const originalSend = res.send;
            res.send = function (body) {
                if (res.statusCode >= 200 && res.statusCode < 300) {
                    const cacheData = {
                        data: typeof body === 'string' ? JSON.parse(body) : body,
                        statusCode: res.statusCode,
                        timestamp: new Date().toISOString(),
                    };
                    redis_1.redisManager.getClient('cache').then(client => client.setEx(cacheKey, config.ttl, JSON.stringify(cacheData))).catch(err => {
                        console.error('Cache storage error:', err);
                    });
                }
                res.setHeader('X-Cache', 'MISS');
                res.setHeader('X-Cache-Key', cacheKey);
                return originalSend.call(this, body);
            };
            next();
        }
        catch (error) {
            console.error('Cache middleware error:', error);
            next();
        }
    };
};
exports.cacheMiddleware = cacheMiddleware;
const cacheInvalidation = (patterns) => {
    return async (req, res, next) => {
        const originalSend = res.send;
        res.send = async function (body) {
            if (res.statusCode >= 200 && res.statusCode < 300) {
                try {
                    const client = await redis_1.redisManager.getClient('cache');
                    for (const pattern of patterns) {
                        if (pattern.startsWith('user:')) {
                            const userId = req.user?.userId;
                            if (userId) {
                                const userPattern = `cache:user:${userId}:*`;
                                await client.eval(`
                  local keys = redis.call('keys', ARGV[1])
                  if #keys > 0 then
                    redis.call('del', unpack(keys))
                  end
                `, 0, userPattern);
                            }
                        }
                        else if (pattern.startsWith('service:')) {
                            const servicePattern = `cache:service:${pattern.substring(8)}*`;
                            await client.eval(`
                local keys = redis.call('keys', ARGV[1])
                if #keys > 0 then
                  redis.call('del', unpack(keys))
                end
              `, 0, servicePattern);
                        }
                        else {
                            await client.del(`cache:${pattern}`);
                        }
                    }
                }
                catch (error) {
                    console.error('Cache invalidation error:', error);
                }
            }
            return originalSend.call(this, body);
        };
        next();
    };
};
exports.cacheInvalidation = cacheInvalidation;
const cacheService = (config) => {
    return function (target, propertyName, descriptor) {
        const method = descriptor.value;
        descriptor.value = async function (...args) {
            const cacheKey = CacheKeyGenerator.generateServiceKey(target.constructor.name, propertyName, args);
            try {
                const client = await redis_1.redisManager.getClient('cache');
                const cachedResult = await client.get(cacheKey);
                if (cachedResult) {
                    return JSON.parse(cachedResult);
                }
                const result = await method.apply(this, args);
                if (result !== undefined && result !== null) {
                    const client = await redis_1.redisManager.getClient('cache');
                    await client.setEx(cacheKey, config.ttl, JSON.stringify(result));
                }
                return result;
            }
            catch (error) {
                console.error('Service cache error:', error);
                return method.apply(this, args);
            }
        };
    };
};
exports.cacheService = cacheService;
class QueryCache {
    static instance;
    constructor() { }
    static getInstance() {
        if (!QueryCache.instance) {
            QueryCache.instance = new QueryCache();
        }
        return QueryCache.instance;
    }
    async get(key) {
        try {
            const client = await redis_1.redisManager.getClient('cache');
            const cached = await client.get(key);
            return cached ? JSON.parse(cached) : null;
        }
        catch (error) {
            console.error('Query cache get error:', error);
            return null;
        }
    }
    async set(key, value, ttl = 300) {
        try {
            const client = await redis_1.redisManager.getClient('cache');
            await client.setEx(key, ttl, JSON.stringify(value));
        }
        catch (error) {
            console.error('Query cache set error:', error);
        }
    }
    async invalidate(pattern) {
        try {
            const client = await redis_1.redisManager.getClient('cache');
            const keys = await client.keys(pattern);
            if (keys.length > 0) {
                await client.del(keys);
            }
        }
        catch (error) {
            console.error('Query cache invalidation error:', error);
        }
    }
    generateKey(collection, query, options) {
        const queryHash = crypto_1.default.createHash('md5').update(JSON.stringify({ query, options })).digest('hex');
        return `cache:db:${collection}:${queryHash}`;
    }
}
exports.QueryCache = QueryCache;
exports.queryCache = QueryCache.getInstance();
class CacheWarmer {
    static instance;
    constructor() { }
    static getInstance() {
        if (!CacheWarmer.instance) {
            CacheWarmer.instance = new CacheWarmer();
        }
        return CacheWarmer.instance;
    }
    async warmCache(patterns, _ttl = 3600) {
        try {
            for (const pattern of patterns) {
                console.log(`Warming cache for pattern: ${pattern}`);
            }
        }
        catch (error) {
            console.error('Cache warming error:', error);
        }
    }
    async preloadUserData(userId, dataTypes) {
        try {
            for (const dataType of dataTypes) {
                console.log(`Preloading ${dataType} for user ${userId}`);
            }
        }
        catch (error) {
            console.error('User data preloading error:', error);
        }
    }
}
exports.CacheWarmer = CacheWarmer;
exports.cacheWarmer = CacheWarmer.getInstance();
class CacheStats {
    static instance;
    stats = new Map();
    constructor() { }
    static getInstance() {
        if (!CacheStats.instance) {
            CacheStats.instance = new CacheStats();
        }
        return CacheStats.instance;
    }
    recordHit(key) {
        const current = this.stats.get(key) || { hits: 0, misses: 0, errors: 0 };
        current.hits++;
        this.stats.set(key, current);
    }
    recordMiss(key) {
        const current = this.stats.get(key) || { hits: 0, misses: 0, errors: 0 };
        current.misses++;
        this.stats.set(key, current);
    }
    recordError(key) {
        const current = this.stats.get(key) || { hits: 0, misses: 0, errors: 0 };
        current.errors++;
        this.stats.set(key, current);
    }
    getStats() {
        const result = {};
        for (const [key, stats] of this.stats.entries()) {
            const total = stats.hits + stats.misses;
            result[key] = {
                ...stats,
                hitRate: total > 0 ? (stats.hits / total) * 100 : 0,
                total,
            };
        }
        return result;
    }
    resetStats() {
        this.stats.clear();
    }
}
exports.CacheStats = CacheStats;
exports.cacheStats = CacheStats.getInstance();
exports.cacheConfigs = {
    short: { ttl: 60 },
    medium: { ttl: 300 },
    long: { ttl: 3600 },
    user: { ttl: 1800 },
    static: { ttl: 86400 },
};
const shortCache = (key) => (0, exports.cacheMiddleware)({ ...exports.cacheConfigs.short, key: key || '' });
exports.shortCache = shortCache;
const mediumCache = (key) => (0, exports.cacheMiddleware)({ ...exports.cacheConfigs.medium, key: key || '' });
exports.mediumCache = mediumCache;
const longCache = (key) => (0, exports.cacheMiddleware)({ ...exports.cacheConfigs.long, key: key || '' });
exports.longCache = longCache;
const userCache = (key) => (0, exports.cacheMiddleware)({ ...exports.cacheConfigs.user, key: key || '' });
exports.userCache = userCache;
const staticCache = (key) => (0, exports.cacheMiddleware)({ ...exports.cacheConfigs.static, key: key || '' });
exports.staticCache = staticCache;
