"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.errorMonitor = exports.ErrorMonitor = exports.requestIdMiddleware = exports.asyncHandler = exports.errorHandler = exports.ConsoleErrorLogger = exports.ExternalServiceError = exports.DatabaseError = exports.RateLimitError = exports.ConflictError = exports.NotFoundError = exports.AuthorizationError = exports.AuthenticationError = exports.BadRequestError = exports.ValidationError = exports.AppError = void 0;
class AppError extends Error {
    statusCode;
    isOperational;
    code;
    details;
    constructor(message, statusCode = 500, isOperational = true, code, details) {
        super(message);
        this.statusCode = statusCode;
        this.isOperational = isOperational;
        this.code = code || '';
        this.details = details;
        Error.captureStackTrace(this, this.constructor);
    }
}
exports.AppError = AppError;
class ValidationError extends AppError {
    constructor(message, details) {
        super(message, 400, true, 'VALIDATION_ERROR', details);
    }
}
exports.ValidationError = ValidationError;
class BadRequestError extends AppError {
    constructor(message, code = 'BAD_REQUEST', details) {
        super(message, 400, true, code, details);
    }
}
exports.BadRequestError = BadRequestError;
class AuthenticationError extends AppError {
    constructor(message = 'Authentication failed') {
        super(message, 401, true, 'AUTHENTICATION_ERROR');
    }
}
exports.AuthenticationError = AuthenticationError;
class AuthorizationError extends AppError {
    constructor(message = 'Access denied') {
        super(message, 403, true, 'AUTHORIZATION_ERROR');
    }
}
exports.AuthorizationError = AuthorizationError;
class NotFoundError extends AppError {
    constructor(message = 'Resource not found') {
        super(message, 404, true, 'NOT_FOUND_ERROR');
    }
}
exports.NotFoundError = NotFoundError;
class ConflictError extends AppError {
    constructor(message = 'Resource conflict') {
        super(message, 409, true, 'CONFLICT_ERROR');
    }
}
exports.ConflictError = ConflictError;
class RateLimitError extends AppError {
    constructor(message = 'Rate limit exceeded') {
        super(message, 429, true, 'RATE_LIMIT_ERROR');
    }
}
exports.RateLimitError = RateLimitError;
class DatabaseError extends AppError {
    constructor(message = 'Database operation failed') {
        super(message, 500, true, 'DATABASE_ERROR');
    }
}
exports.DatabaseError = DatabaseError;
class ExternalServiceError extends AppError {
    constructor(message = 'External service error') {
        super(message, 502, true, 'EXTERNAL_SERVICE_ERROR');
    }
}
exports.ExternalServiceError = ExternalServiceError;
class ConsoleErrorLogger {
    error(error, context) {
        console.error('ERROR:', {
            message: error.message,
            stack: error.stack,
            context,
            timestamp: new Date().toISOString(),
        });
    }
    warn(message, context) {
        console.warn('WARN:', {
            message,
            context,
            timestamp: new Date().toISOString(),
        });
    }
    info(message, context) {
        console.info('INFO:', {
            message,
            context,
            timestamp: new Date().toISOString(),
        });
    }
}
exports.ConsoleErrorLogger = ConsoleErrorLogger;
const errorHandler = (logger = new ConsoleErrorLogger()) => {
    return (err, req, res, _next) => {
        const requestId = req.headers['x-request-id'] || req.headers['x-correlation-id'] || 'unknown';
        logger.error(err, {
            requestId,
            url: req.url,
            method: req.method,
            userAgent: req.get('User-Agent'),
            ip: req.ip,
            userId: req.user?.userId,
        });
        if (res.headersSent) {
            return;
        }
        if (err instanceof AppError) {
            res.status(err.statusCode).json({
                success: false,
                error: {
                    code: err.code,
                    message: err.message,
                    ...(process.env['NODE_ENV'] === 'development' && { details: err.details }),
                    ...(process.env['NODE_ENV'] === 'development' && { stack: err.stack }),
                },
                requestId,
                timestamp: new Date().toISOString(),
            });
            return;
        }
        if (err.name === 'ValidationError') {
            res.status(400).json({
                success: false,
                error: {
                    code: 'VALIDATION_ERROR',
                    message: 'Validation failed',
                    details: err.message,
                },
                requestId,
                timestamp: new Date().toISOString(),
            });
            return;
        }
        if (err.name === 'JsonWebTokenError') {
            res.status(401).json({
                success: false,
                error: {
                    code: 'INVALID_TOKEN',
                    message: 'Invalid token provided',
                },
                requestId,
                timestamp: new Date().toISOString(),
            });
            return;
        }
        if (err.name === 'TokenExpiredError') {
            res.status(401).json({
                success: false,
                error: {
                    code: 'TOKEN_EXPIRED',
                    message: 'Token has expired',
                },
                requestId,
                timestamp: new Date().toISOString(),
            });
            return;
        }
        if (err.name === 'MongoError' || err.name === 'MongoServerError') {
            const mongoError = err;
            if (mongoError.code === 11000) {
                res.status(409).json({
                    success: false,
                    error: {
                        code: 'DUPLICATE_KEY',
                        message: 'Duplicate key error',
                        details: mongoError.keyValue,
                    },
                    requestId,
                    timestamp: new Date().toISOString(),
                });
                return;
            }
        }
        const statusCode = 500;
        const message = process.env['NODE_ENV'] === 'production'
            ? 'Internal server error'
            : err.message;
        res.status(statusCode).json({
            success: false,
            error: {
                code: 'INTERNAL_SERVER_ERROR',
                message,
                ...(process.env['NODE_ENV'] === 'development' && { stack: err.stack }),
            },
            requestId,
            timestamp: new Date().toISOString(),
        });
    };
};
exports.errorHandler = errorHandler;
const asyncHandler = (fn) => {
    return (req, res, next) => {
        Promise.resolve(fn(req, res, next)).catch(next);
    };
};
exports.asyncHandler = asyncHandler;
const requestIdMiddleware = (req, res, next) => {
    const requestId = req.headers['x-request-id'] ||
        req.headers['x-correlation-id'] ||
        `req_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    req.headers['x-request-id'] = requestId;
    res.setHeader('x-request-id', requestId);
    next();
};
exports.requestIdMiddleware = requestIdMiddleware;
class ErrorMonitor {
    static instance;
    errorCounts = new Map();
    alertThreshold = 10;
    static getInstance() {
        if (!ErrorMonitor.instance) {
            ErrorMonitor.instance = new ErrorMonitor();
        }
        return ErrorMonitor.instance;
    }
    trackError(error) {
        const errorKey = error.code || error.constructor.name;
        const currentCount = this.errorCounts.get(errorKey) || 0;
        const newCount = currentCount + 1;
        this.errorCounts.set(errorKey, newCount);
        if (newCount >= this.alertThreshold) {
            this.sendAlert(errorKey, newCount);
        }
    }
    sendAlert(errorKey, count) {
        console.error(`🚨 ALERT: ${errorKey} occurred ${count} times`);
    }
    getErrorStats() {
        return Object.fromEntries(this.errorCounts);
    }
    resetCounts() {
        this.errorCounts.clear();
    }
}
exports.ErrorMonitor = ErrorMonitor;
exports.errorMonitor = ErrorMonitor.getInstance();
