"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.cleanupBackups = exports.listBackups = exports.executeDRRunbook = exports.restoreBackup = exports.createBackup = exports.backupManager = exports.BackupManager = void 0;
const promises_1 = __importDefault(require("fs/promises"));
const path_1 = __importDefault(require("path"));
const child_process_1 = require("child_process");
const crypto_1 = __importDefault(require("crypto"));
const util_1 = require("util");
const redis_1 = require("../../config/redis");
const env_1 = require("../../config/env");
const secrets_manager_1 = require("../security/secrets-manager");
const execAsync = (0, util_1.promisify)(child_process_1.exec);
class BackupManager {
    static instance;
    backupConfigs = new Map();
    backupHistory = new Map();
    drRunbooks = new Map();
    constructor() {
        this.initializeBackupConfigs();
        this.initializeDRRunbooks();
    }
    static getInstance() {
        if (!BackupManager.instance) {
            BackupManager.instance = new BackupManager();
        }
        return BackupManager.instance;
    }
    initializeBackupConfigs() {
        const configs = [
            {
                id: 'db-daily',
                name: 'Daily Database Backup',
                type: 'full',
                source: 'database',
                schedule: 'daily',
                retention: 30,
                compression: true,
                encryption: true,
                destination: 'local',
                enabled: true,
            },
            {
                id: 'redis-hourly',
                name: 'Hourly Redis Backup',
                type: 'incremental',
                source: 'redis',
                schedule: 'hourly',
                retention: 7,
                compression: true,
                encryption: false,
                destination: 'local',
                enabled: true,
            },
            {
                id: 'secrets-weekly',
                name: 'Weekly Secrets Backup',
                type: 'full',
                source: 'secrets',
                schedule: 'weekly',
                retention: 90,
                compression: true,
                encryption: true,
                destination: 'local',
                enabled: true,
            },
            {
                id: 'logs-daily',
                name: 'Daily Logs Backup',
                type: 'full',
                source: 'logs',
                schedule: 'daily',
                retention: 90,
                compression: true,
                encryption: false,
                destination: 'local',
                enabled: true,
            },
        ];
        configs.forEach(config => {
            this.backupConfigs.set(config.id, config);
        });
    }
    initializeDRRunbooks() {
        const runbooks = [
            {
                id: 'database-recovery',
                name: 'Database Recovery Procedure',
                description: 'Recover database from backup in case of corruption or data loss',
                severity: 'critical',
                estimatedTime: 30,
                dependencies: ['backup-verification'],
                contacts: ['dba@company.com', 'devops@company.com'],
                steps: [
                    {
                        id: 'stop-app',
                        name: 'Stop Application',
                        description: 'Stop the application to prevent data corruption',
                        command: 'pm2 stop all',
                        order: 1,
                        timeout: 60,
                    },
                    {
                        id: 'verify-backup',
                        name: 'Verify Latest Backup',
                        description: 'Verify the integrity of the latest backup',
                        command: 'mongodump --verify',
                        order: 2,
                        timeout: 300,
                    },
                    {
                        id: 'restore-db',
                        name: 'Restore Database',
                        description: 'Restore database from backup',
                        command: 'mongorestore --drop',
                        order: 3,
                        timeout: 1800,
                    },
                    {
                        id: 'verify-restore',
                        name: 'Verify Restore',
                        description: 'Verify that the restore was successful',
                        command: 'mongo --eval "db.stats()"',
                        order: 4,
                        timeout: 120,
                    },
                    {
                        id: 'start-app',
                        name: 'Start Application',
                        description: 'Start the application',
                        command: 'pm2 start all',
                        order: 5,
                        timeout: 60,
                    },
                ],
            },
            {
                id: 'redis-recovery',
                name: 'Redis Recovery Procedure',
                description: 'Recover Redis data from backup',
                severity: 'high',
                estimatedTime: 15,
                dependencies: [],
                contacts: ['devops@company.com'],
                steps: [
                    {
                        id: 'stop-redis',
                        name: 'Stop Redis',
                        description: 'Stop Redis service',
                        command: 'systemctl stop redis',
                        order: 1,
                        timeout: 30,
                    },
                    {
                        id: 'restore-redis',
                        name: 'Restore Redis Data',
                        description: 'Restore Redis data from backup',
                        command: 'redis-cli --rdb /backup/redis.rdb',
                        order: 2,
                        timeout: 300,
                    },
                    {
                        id: 'start-redis',
                        name: 'Start Redis',
                        description: 'Start Redis service',
                        command: 'systemctl start redis',
                        order: 3,
                        timeout: 30,
                    },
                ],
            },
            {
                id: 'full-system-recovery',
                name: 'Full System Recovery',
                description: 'Complete system recovery from disaster',
                severity: 'critical',
                estimatedTime: 120,
                dependencies: ['database-recovery', 'redis-recovery'],
                contacts: ['cto@company.com', 'devops@company.com'],
                steps: [
                    {
                        id: 'assess-damage',
                        name: 'Assess System Damage',
                        description: 'Assess the extent of system damage',
                        manual: true,
                        order: 1,
                        timeout: 600,
                    },
                    {
                        id: 'backup-current',
                        name: 'Backup Current State',
                        description: 'Create backup of current system state',
                        command: './scripts/backup-current.sh',
                        order: 2,
                        timeout: 1800,
                    },
                    {
                        id: 'restore-infrastructure',
                        name: 'Restore Infrastructure',
                        description: 'Restore server infrastructure',
                        manual: true,
                        order: 3,
                        timeout: 3600,
                    },
                    {
                        id: 'restore-database',
                        name: 'Restore Database',
                        description: 'Execute database recovery procedure',
                        script: 'database-recovery',
                        order: 4,
                        timeout: 1800,
                    },
                    {
                        id: 'restore-redis',
                        name: 'Restore Redis',
                        description: 'Execute Redis recovery procedure',
                        script: 'redis-recovery',
                        order: 5,
                        timeout: 900,
                    },
                    {
                        id: 'verify-system',
                        name: 'Verify System',
                        description: 'Verify all systems are operational',
                        command: './scripts/health-check.sh',
                        order: 6,
                        timeout: 300,
                    },
                ],
            },
        ];
        runbooks.forEach(runbook => {
            this.drRunbooks.set(runbook.id, runbook);
        });
    }
    async createBackup(configId) {
        const config = this.backupConfigs.get(configId);
        if (!config) {
            throw new Error(`Backup configuration ${configId} not found`);
        }
        const backupId = `${configId}-${Date.now()}`;
        const startTime = Date.now();
        const metadata = {
            id: backupId,
            configId,
            timestamp: new Date(),
            size: 0,
            checksum: '',
            status: 'in_progress',
            duration: 0,
            files: [],
        };
        try {
            console.log(`🔄 Starting backup: ${config.name}`);
            switch (config.source) {
                case 'database':
                    await this.backupDatabase(config, metadata);
                    break;
                case 'redis':
                    await this.backupRedis(config, metadata);
                    break;
                case 'secrets':
                    await this.backupSecrets(config, metadata);
                    break;
                case 'logs':
                    await this.backupLogs(config, metadata);
                    break;
                default:
                    throw new Error(`Unsupported backup source: ${config.source}`);
            }
            metadata.status = 'completed';
            metadata.duration = Date.now() - startTime;
            await this.storeBackupMetadata(metadata);
            console.log(`✅ Backup completed: ${config.name} (${metadata.duration}ms)`);
            return metadata;
        }
        catch (error) {
            metadata.status = 'failed';
            metadata.error = error instanceof Error ? error.message : 'Unknown error';
            metadata.duration = Date.now() - startTime;
            await this.storeBackupMetadata(metadata);
            console.error(`❌ Backup failed: ${config.name}`, error);
            throw error;
        }
    }
    async backupDatabase(_config, metadata) {
        const backupDir = path_1.default.join(process.cwd(), 'backups', 'database');
        await promises_1.default.mkdir(backupDir, { recursive: true });
        const filename = `db-backup-${Date.now()}.gz`;
        const filepath = path_1.default.join(backupDir, filename);
        const dumpCmd = process.platform === 'win32' ? 'where mongodump' : 'which mongodump';
        try {
            await execAsync(dumpCmd);
        }
        catch {
            console.warn('⚠️ Skipping DB backup: mongodump not found in PATH');
            metadata.status = 'completed';
            metadata.duration = 0;
            return;
        }
        await execAsync(`mongodump --uri="${env_1.env.MONGODB_URI}" --archive="${filepath}" --gzip`);
        const stats = await promises_1.default.stat(filepath);
        metadata.size = stats.size;
        metadata.files.push(filepath);
        const fileBuffer = await promises_1.default.readFile(filepath);
        const hash = crypto_1.default.createHash('sha256').update(fileBuffer).digest('hex');
        metadata.checksum = hash;
    }
    async backupRedis(_config, metadata) {
        const backupDir = path_1.default.join(process.cwd(), 'backups', 'redis');
        await promises_1.default.mkdir(backupDir, { recursive: true });
        const filename = `redis-backup-${Date.now()}.rdb`;
        const filepath = path_1.default.join(backupDir, filename);
        const client = await redis_1.redisManager.getClient('main');
        await client.save();
        await execAsync(`cp /var/lib/redis/dump.rdb "${filepath}"`);
        const stats = await promises_1.default.stat(filepath);
        metadata.size = stats.size;
        metadata.files.push(filepath);
        const { stdout: checksum } = await execAsync(`sha256sum "${filepath}"`);
        metadata.checksum = checksum?.split(' ')[0] || '';
    }
    async backupSecrets(_config, metadata) {
        const backupDir = path_1.default.join(process.cwd(), 'backups', 'secrets');
        await promises_1.default.mkdir(backupDir, { recursive: true });
        const filename = `secrets-backup-${Date.now()}.json`;
        const filepath = path_1.default.join(backupDir, filename);
        const secrets = await secrets_manager_1.secretsManager.listSecrets();
        const backupData = {
            timestamp: new Date().toISOString(),
            secrets: secrets,
            version: '1.0',
        };
        await promises_1.default.writeFile(filepath, JSON.stringify(backupData, null, 2));
        const stats = await promises_1.default.stat(filepath);
        metadata.size = stats.size;
        metadata.files.push(filepath);
        const { stdout: checksum } = await execAsync(`sha256sum "${filepath}"`);
        metadata.checksum = checksum?.split(' ')[0] || '';
    }
    async backupLogs(_config, metadata) {
        const backupDir = path_1.default.join(process.cwd(), 'backups', 'logs');
        await promises_1.default.mkdir(backupDir, { recursive: true });
        const filename = `logs-backup-${Date.now()}.tar.gz`;
        const filepath = path_1.default.join(backupDir, filename);
        await execAsync(`tar -czf "${filepath}" -C logs .`);
        const stats = await promises_1.default.stat(filepath);
        metadata.size = stats.size;
        metadata.files.push(filepath);
        const { stdout: checksum } = await execAsync(`sha256sum "${filepath}"`);
        metadata.checksum = checksum?.split(' ')[0] || '';
    }
    async restoreFromBackup(backupId, config) {
        const metadata = await this.getBackupMetadata(backupId);
        if (!metadata) {
            throw new Error(`Backup ${backupId} not found`);
        }
        const backupConfig = this.backupConfigs.get(metadata.configId);
        if (!backupConfig) {
            throw new Error(`Backup configuration not found`);
        }
        console.log(`🔄 Starting restore from backup: ${backupId}`);
        try {
            switch (backupConfig.source) {
                case 'database':
                    await this.restoreDatabase(metadata, config);
                    break;
                case 'redis':
                    await this.restoreRedis(metadata, config);
                    break;
                case 'secrets':
                    await this.restoreSecrets(metadata, config);
                    break;
                case 'logs':
                    await this.restoreLogs(metadata, config);
                    break;
                default:
                    throw new Error(`Unsupported restore source: ${backupConfig.source}`);
            }
            console.log(`✅ Restore completed: ${backupId}`);
        }
        catch (error) {
            console.error(`❌ Restore failed: ${backupId}`, error);
            throw error;
        }
    }
    async restoreDatabase(metadata, _config) {
        const filepath = metadata.files[0];
        if (!filepath) {
            throw new Error('No backup file found');
        }
        const { stdout: checksum } = await execAsync(`sha256sum "${filepath}"`);
        const currentChecksum = checksum.split(' ')[0];
        if (currentChecksum !== metadata.checksum) {
            throw new Error('Backup file checksum verification failed');
        }
        await execAsync(`mongorestore --uri="${env_1.env.MONGODB_URI}" --archive="${filepath}" --gzip --drop`);
    }
    async restoreRedis(metadata, _config) {
        const filepath = metadata.files[0];
        if (!filepath) {
            throw new Error('No backup file found');
        }
        await execAsync('systemctl stop redis');
        await execAsync(`cp "${filepath}" /var/lib/redis/dump.rdb`);
        await execAsync('systemctl start redis');
    }
    async restoreSecrets(metadata, _config) {
        const filepath = metadata.files[0];
        if (!filepath) {
            throw new Error('No backup file found');
        }
        const backupData = JSON.parse(await promises_1.default.readFile(filepath, 'utf8'));
        for (const secret of backupData.secrets) {
            await secrets_manager_1.secretsManager.createSecret(secret.name, secret.value, secret.type, secret.environment, secret.rotationPeriod);
        }
    }
    async restoreLogs(metadata, _config) {
        const filepath = metadata.files[0];
        if (!filepath) {
            throw new Error('No backup file found');
        }
        await execAsync(`tar -xzf "${filepath}" -C logs`);
    }
    async executeDRRunbook(runbookId) {
        const runbook = this.drRunbooks.get(runbookId);
        if (!runbook) {
            throw new Error(`DR runbook ${runbookId} not found`);
        }
        console.log(`🚨 Executing DR runbook: ${runbook.name}`);
        for (const step of runbook.steps.sort((a, b) => a.order - b.order)) {
            console.log(`📋 Executing step: ${step.name}`);
            try {
                if (step.command) {
                    await execAsync(step.command, { timeout: step.timeout * 1000 });
                }
                else if (step.script) {
                    await this.executeDRRunbook(step.script);
                }
                else if (step.manual) {
                    console.log(`⏸️ Manual step required: ${step.description}`);
                }
                console.log(`✅ Step completed: ${step.name}`);
            }
            catch (error) {
                console.error(`❌ Step failed: ${step.name}`, error);
                if (step.rollback) {
                    console.log(`🔄 Executing rollback for step: ${step.name}`);
                    await execAsync(step.rollback);
                }
                throw error;
            }
        }
        console.log(`✅ DR runbook completed: ${runbook.name}`);
    }
    async listBackups(configId) {
        const backups = [];
        for (const backupList of this.backupHistory.values()) {
            for (const backup of backupList) {
                if (!configId || backup.configId === configId) {
                    backups.push(backup);
                }
            }
        }
        return backups.sort((a, b) => b.timestamp.getTime() - a.timestamp.getTime());
    }
    async getBackupMetadata(backupId) {
        for (const backupList of this.backupHistory.values()) {
            const backup = backupList.find(b => b.id === backupId);
            if (backup)
                return backup;
        }
        return null;
    }
    async storeBackupMetadata(metadata) {
        if (!this.backupHistory.has(metadata.configId)) {
            this.backupHistory.set(metadata.configId, []);
        }
        this.backupHistory.get(metadata.configId).push(metadata);
        try {
            const client = await redis_1.redisManager.getClient('main');
            const key = `backup:${metadata.configId}:${metadata.id}`;
            await client.setEx(key, 86400 * 30, JSON.stringify(metadata));
        }
        catch (error) {
            console.error('Failed to store backup metadata in Redis:', error);
        }
    }
    async cleanupOldBackups() {
        const cleanedBackups = [];
        for (const [configId, backupList] of this.backupHistory.entries()) {
            const config = this.backupConfigs.get(configId);
            if (!config)
                continue;
            const cutoffDate = new Date(Date.now() - config.retention * 24 * 60 * 60 * 1000);
            const oldBackups = backupList.filter(backup => backup.timestamp < cutoffDate);
            for (const backup of oldBackups) {
                for (const file of backup.files) {
                    try {
                        await promises_1.default.unlink(file);
                        console.log(`🗑️ Deleted old backup file: ${file}`);
                    }
                    catch (error) {
                        console.error(`Failed to delete backup file: ${file}`, error);
                    }
                }
                cleanedBackups.push(backup.id);
            }
            this.backupHistory.set(configId, backupList.filter(backup => backup.timestamp >= cutoffDate));
        }
        return cleanedBackups;
    }
    getDRRunbooks() {
        return Array.from(this.drRunbooks.values());
    }
    getDRRunbook(runbookId) {
        return this.drRunbooks.get(runbookId) || null;
    }
}
exports.BackupManager = BackupManager;
exports.backupManager = BackupManager.getInstance();
const createBackup = (configId) => exports.backupManager.createBackup(configId);
exports.createBackup = createBackup;
const restoreBackup = (backupId, config) => exports.backupManager.restoreFromBackup(backupId, config);
exports.restoreBackup = restoreBackup;
const executeDRRunbook = (runbookId) => exports.backupManager.executeDRRunbook(runbookId);
exports.executeDRRunbook = executeDRRunbook;
const listBackups = (configId) => exports.backupManager.listBackups(configId);
exports.listBackups = listBackups;
const cleanupBackups = () => exports.backupManager.cleanupOldBackups();
exports.cleanupBackups = cleanupBackups;
