"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateComplianceReport = exports.checkConsent = exports.recordConsent = exports.scanDLP = exports.detectPII = exports.complianceManager = exports.ComplianceManager = exports.DataClassification = void 0;
const crypto_1 = __importDefault(require("crypto"));
const redis_1 = require("../../config/redis");
const env_1 = require("../../config/env");
var DataClassification;
(function (DataClassification) {
    DataClassification["PUBLIC"] = "public";
    DataClassification["INTERNAL"] = "internal";
    DataClassification["CONFIDENTIAL"] = "confidential";
    DataClassification["RESTRICTED"] = "restricted";
})(DataClassification || (exports.DataClassification = DataClassification = {}));
class ComplianceManager {
    static instance;
    piiFields = new Map();
    retentionPolicies = new Map();
    dlpRules = new Map();
    consentRecords = new Map();
    constructor() {
        this.initializePIIFields();
        this.initializeRetentionPolicies();
        this.initializeDLPRules();
    }
    static getInstance() {
        if (!ComplianceManager.instance) {
            ComplianceManager.instance = new ComplianceManager();
        }
        return ComplianceManager.instance;
    }
    initializePIIFields() {
        const piiConfigs = [
            {
                field: 'email',
                type: 'email',
                sensitivity: 'medium',
                encryption: true,
                masking: false,
                retention: 2555,
            },
            {
                field: 'phone',
                type: 'phone',
                sensitivity: 'high',
                encryption: true,
                masking: true,
                retention: 2555,
            },
            {
                field: 'bvn',
                type: 'bvn',
                sensitivity: 'high',
                encryption: true,
                masking: true,
                retention: 3650,
            },
            {
                field: 'nin',
                type: 'nin',
                sensitivity: 'high',
                encryption: true,
                masking: true,
                retention: 3650,
            },
            {
                field: 'address',
                type: 'address',
                sensitivity: 'medium',
                encryption: true,
                masking: false,
                retention: 2555,
            },
            {
                field: 'bank_account',
                type: 'bank_account',
                sensitivity: 'high',
                encryption: true,
                masking: true,
                retention: 3650,
            },
            {
                field: 'card_number',
                type: 'card_number',
                sensitivity: 'high',
                encryption: true,
                masking: true,
                retention: 1825,
            },
        ];
        piiConfigs.forEach(config => {
            this.piiFields.set(config.field, config);
        });
    }
    initializeRetentionPolicies() {
        const policies = [
            {
                dataType: 'user_profile',
                retentionPeriod: 2555,
                disposalMethod: 'anonymize',
                legalBasis: 'NDPR Section 2.1 - Legitimate Interest',
                exceptions: ['ongoing_legal_proceedings'],
            },
            {
                dataType: 'transaction_records',
                retentionPeriod: 3650,
                disposalMethod: 'archive',
                legalBasis: 'CBN Guidelines - Financial Records',
                exceptions: ['audit_requirements'],
            },
            {
                dataType: 'kyc_documents',
                retentionPeriod: 3650,
                disposalMethod: 'delete',
                legalBasis: 'NDPR Section 2.1 - Consent',
                exceptions: ['regulatory_requirements'],
            },
            {
                dataType: 'payment_data',
                retentionPeriod: 1825,
                disposalMethod: 'anonymize',
                legalBasis: 'PCI DSS Requirements',
                exceptions: ['dispute_resolution'],
            },
        ];
        policies.forEach(policy => {
            this.retentionPolicies.set(policy.dataType, policy);
        });
    }
    initializeDLPRules() {
        const rules = [
            {
                id: 'bvn_pattern',
                name: 'BVN Pattern Detection',
                pattern: /\b\d{11}\b/g,
                action: 'encrypt',
                severity: 'high',
                description: 'Detects and encrypts BVN numbers',
                enabled: true,
            },
            {
                id: 'nin_pattern',
                name: 'NIN Pattern Detection',
                pattern: /\b\d{11}\b/g,
                action: 'encrypt',
                severity: 'high',
                description: 'Detects and encrypts NIN numbers',
                enabled: true,
            },
            {
                id: 'bank_account_pattern',
                name: 'Bank Account Pattern Detection',
                pattern: /\b\d{10}\b/g,
                action: 'mask',
                severity: 'medium',
                description: 'Detects and masks bank account numbers',
                enabled: true,
            },
            {
                id: 'card_number_pattern',
                name: 'Card Number Pattern Detection',
                pattern: /\b\d{4}[\s-]?\d{4}[\s-]?\d{4}[\s-]?\d{4}\b/g,
                action: 'mask',
                severity: 'high',
                description: 'Detects and masks credit card numbers',
                enabled: true,
            },
            {
                id: 'email_pattern',
                name: 'Email Pattern Detection',
                pattern: /\b[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Z|a-z]{2,}\b/g,
                action: 'log',
                severity: 'low',
                description: 'Logs email addresses for compliance',
                enabled: true,
            },
        ];
        rules.forEach(rule => {
            this.dlpRules.set(rule.id, rule);
        });
    }
    detectAndProcessPII(data) {
        const processed = { ...data };
        for (const [field, config] of this.piiFields) {
            if (processed[field]) {
                processed[field] = this.processPIIField(processed[field], config);
            }
        }
        return processed;
    }
    processPIIField(value, config) {
        if (config.encryption) {
            value = this.encryptPII(value);
        }
        if (config.masking) {
            value = this.maskPII(value, config.type);
        }
        return value;
    }
    encryptPII(value) {
        const algorithm = 'aes-256-gcm';
        const key = crypto_1.default.scryptSync(env_1.env.JWT_SECRET, 'salt', 32);
        const iv = crypto_1.default.randomBytes(16);
        const cipher = crypto_1.default.createCipheriv(algorithm, key, iv);
        let encrypted = cipher.update(value, 'utf8', 'hex');
        encrypted += cipher.final('hex');
        const authTag = cipher.getAuthTag();
        return `encrypted:${iv.toString('hex')}:${authTag.toString('hex')}:${encrypted}`;
    }
    maskPII(value, type) {
        switch (type) {
            case 'phone':
                return value.replace(/(\d{3})(\d{3})(\d{4})/, '***-***-$3');
            case 'bvn':
            case 'nin':
                return value.replace(/(\d{3})(\d{3})(\d{5})/, '***-***-$3');
            case 'bank_account':
                return value.replace(/(\d{3})(\d{3})(\d{4})/, '***-***-$3');
            case 'card_number':
                return value.replace(/(\d{4})(\d{4})(\d{4})(\d{4})/, '****-****-****-$4');
            default:
                return value;
        }
    }
    scanForSensitiveData(content) {
        const triggeredRules = [];
        for (const rule of this.dlpRules.values()) {
            if (rule.enabled && rule.pattern.test(content)) {
                triggeredRules.push(rule);
            }
        }
        return triggeredRules;
    }
    applyDLPActions(content, rules) {
        let processedContent = content;
        for (const rule of rules) {
            switch (rule.action) {
                case 'block':
                    throw new Error(`Content blocked by DLP rule: ${rule.name}`);
                case 'encrypt':
                    processedContent = processedContent.replace(rule.pattern, (match) => this.encryptPII(match));
                    break;
                case 'mask':
                    processedContent = processedContent.replace(rule.pattern, (match) => this.maskPII(match, 'card_number'));
                    break;
                case 'log':
                    this.logDLPEvent(rule, content);
                    break;
            }
        }
        return processedContent;
    }
    logDLPEvent(rule, content) {
        console.warn(`🔒 DLP Event: ${rule.name} triggered`, {
            timestamp: new Date().toISOString(),
            rule: rule.name,
            severity: rule.severity,
            contentLength: content.length,
        });
    }
    async recordConsent(userId, purpose, consentType, dataUsage, expiresAt) {
        const consent = {
            id: crypto_1.default.randomUUID(),
            userId,
            purpose,
            consentType,
            grantedAt: new Date(),
            dataUsage,
            version: '1.0',
            ...(expiresAt && { expiresAt }),
        };
        if (!this.consentRecords.has(userId)) {
            this.consentRecords.set(userId, []);
        }
        this.consentRecords.get(userId).push(consent);
        await this.storeConsentInRedis(consent);
        return consent;
    }
    async withdrawConsent(userId, purpose) {
        const userConsents = this.consentRecords.get(userId) || [];
        const consent = userConsents.find(c => c.purpose === purpose && c.consentType !== 'withdrawn');
        if (consent) {
            consent.consentType = 'withdrawn';
            consent.withdrawnAt = new Date();
            await this.storeConsentInRedis(consent);
        }
    }
    async hasValidConsent(userId, purpose) {
        const userConsents = this.consentRecords.get(userId) || [];
        const consent = userConsents.find(c => c.purpose === purpose);
        if (!consent)
            return false;
        if (consent.consentType === 'withdrawn')
            return false;
        if (consent.expiresAt && consent.expiresAt < new Date())
            return false;
        return true;
    }
    async checkRetentionCompliance(dataType, createdAt) {
        const policy = this.retentionPolicies.get(dataType);
        if (!policy) {
            return {
                compliant: true,
                daysUntilDisposal: 0,
                disposalMethod: 'none',
            };
        }
        const daysSinceCreation = (Date.now() - createdAt.getTime()) / (1000 * 60 * 60 * 24);
        const daysUntilDisposal = Math.max(0, policy.retentionPeriod - daysSinceCreation);
        return {
            compliant: daysUntilDisposal > 0,
            daysUntilDisposal: Math.ceil(daysUntilDisposal),
            disposalMethod: policy.disposalMethod,
        };
    }
    async disposeData(dataType, _data) {
        const policy = this.retentionPolicies.get(dataType);
        if (!policy)
            return;
        switch (policy.disposalMethod) {
            case 'delete':
                console.log(`🗑️ Data marked for deletion: ${dataType}`);
                break;
            case 'archive':
                console.log(`📦 Data archived: ${dataType}`);
                break;
            case 'anonymize':
                console.log(`👤 Data anonymized: ${dataType}`);
                break;
        }
    }
    async generateComplianceReport() {
        const report = {
            ndprCompliance: env_1.env.NDPR_COMPLIANCE,
            cbnCompliance: env_1.env.CBN_COMPLIANCE,
            piiProtection: this.piiFields.size > 0,
            retentionCompliance: this.retentionPolicies.size > 0,
            consentCompliance: this.consentRecords.size > 0,
            details: {
                piiFieldsCount: this.piiFields.size,
                retentionPoliciesCount: this.retentionPolicies.size,
                dlpRulesCount: this.dlpRules.size,
                activeConsentsCount: Array.from(this.consentRecords.values()).flat().length,
                lastAudit: new Date().toISOString(),
            },
        };
        return report;
    }
    async storeConsentInRedis(consent) {
        try {
            const client = await redis_1.redisManager.getClient('main');
            const key = `consent:${consent.userId}:${consent.purpose}`;
            await client.setEx(key, 86400 * 365, JSON.stringify(consent));
        }
        catch (error) {
            console.error('Failed to store consent in Redis:', error);
        }
    }
}
exports.ComplianceManager = ComplianceManager;
exports.complianceManager = ComplianceManager.getInstance();
const detectPII = (data) => exports.complianceManager.detectAndProcessPII(data);
exports.detectPII = detectPII;
const scanDLP = (content) => exports.complianceManager.scanForSensitiveData(content);
exports.scanDLP = scanDLP;
const recordConsent = (userId, purpose, consentType, dataUsage, expiresAt) => exports.complianceManager.recordConsent(userId, purpose, consentType, dataUsage, expiresAt);
exports.recordConsent = recordConsent;
const checkConsent = (userId, purpose) => exports.complianceManager.hasValidConsent(userId, purpose);
exports.checkConsent = checkConsent;
const generateComplianceReport = () => exports.complianceManager.generateComplianceReport();
exports.generateComplianceReport = generateComplianceReport;
