#!/usr/bin/env ts-node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateEnvironment = void 0;
const dotenv_1 = require("dotenv");
const zod_1 = require("zod");
(0, dotenv_1.config)();
const requiredEnvSchema = zod_1.z.object({
    NODE_ENV: zod_1.z.enum(['development', 'production', 'test']).default('development'),
    PORT: zod_1.z.string().pipe(zod_1.z.coerce.number().min(1, 'Port must be a positive number')),
    API_VERSION: zod_1.z.string().min(1, 'API version is required'),
    MONGODB_URI: zod_1.z.string().url('MongoDB URI must be a valid URL'),
    JWT_SECRET: zod_1.z.string().min(32, 'JWT secret must be at least 32 characters'),
    JWT_EXPIRES_IN: zod_1.z.string().min(1, 'JWT expiration is required'),
    JWT_REFRESH_EXPIRES_IN: zod_1.z.string().min(1, 'JWT refresh expiration is required'),
    REDIS_URL: zod_1.z.string().url('Redis URL must be a valid URL'),
    PAYSTACK_SECRET_KEY: zod_1.z.string().min(1, 'Paystack secret key is required'),
    PAYSTACK_PUBLIC_KEY: zod_1.z.string().min(1, 'Paystack public key is required'),
    PAYSTACK_WEBHOOK_SECRET: zod_1.z.string().min(1, 'Paystack webhook secret is required'),
    FLUTTERWAVE_SECRET_KEY: zod_1.z.string().min(1, 'Flutterwave secret key is required'),
    FLUTTERWAVE_PUBLIC_KEY: zod_1.z.string().min(1, 'Flutterwave public key is required'),
    NIMC_API_KEY: zod_1.z.string().min(1, 'NIMC API key is required'),
    NIMC_API_URL: zod_1.z.string().url('NIMC API URL must be a valid URL'),
    BVN_API_KEY: zod_1.z.string().min(1, 'BVN API key is required'),
    BVN_API_URL: zod_1.z.string().url('BVN API URL must be a valid URL'),
    BCRYPT_ROUNDS: zod_1.z.string().pipe(zod_1.z.coerce.number().min(10, 'BCrypt rounds must be at least 10')),
    RATE_LIMIT_WINDOW_MS: zod_1.z.string().pipe(zod_1.z.coerce.number().min(1000, 'Rate limit window must be at least 1000ms')),
    RATE_LIMIT_MAX_REQUESTS: zod_1.z.string().pipe(zod_1.z.coerce.number().min(1, 'Rate limit max requests must be at least 1')),
    LOG_LEVEL: zod_1.z.enum(['error', 'warn', 'info', 'debug']).default('info'),
    LOG_PRETTY_PRINT: zod_1.z.string().pipe(zod_1.z.coerce.boolean()).default('true'),
    CORS_ORIGIN: zod_1.z.string().min(1, 'CORS origin is required'),
    CORS_CREDENTIALS: zod_1.z.string().pipe(zod_1.z.coerce.boolean()).default('true'),
    MAX_FILE_SIZE: zod_1.z.string().pipe(zod_1.z.coerce.number().min(1024, 'Max file size must be at least 1KB')),
    UPLOAD_PATH: zod_1.z.string().min(1, 'Upload path is required'),
    SMTP_HOST: zod_1.z.string().min(1, 'SMTP host is required'),
    SMTP_PORT: zod_1.z.string().pipe(zod_1.z.coerce.number().min(1).max(65535, 'SMTP port must be between 1 and 65535')),
    SMTP_USER: zod_1.z.string().email('SMTP user must be a valid email'),
    SMTP_PASS: zod_1.z.string().min(1, 'SMTP password is required'),
    SMS_API_KEY: zod_1.z.string().min(1, 'SMS API key is required'),
    SMS_API_URL: zod_1.z.string().url('SMS API URL must be a valid URL'),
    APP_NAME: zod_1.z.string().min(1, 'App name is required'),
    APP_URL: zod_1.z.string().url('App URL must be a valid URL'),
    FRONTEND_URL: zod_1.z.string().url('Frontend URL must be a valid URL'),
    NDPR_COMPLIANCE: zod_1.z.string().pipe(zod_1.z.coerce.boolean()).default('true'),
    CBN_COMPLIANCE: zod_1.z.string().pipe(zod_1.z.coerce.boolean()).default('true'),
    DATA_RETENTION_DAYS: zod_1.z.string().pipe(zod_1.z.coerce.number().min(1, 'Data retention days must be at least 1')),
});
const validateEnvironment = () => {
    console.log('🔍 Validating environment configuration...\n');
    try {
        const validatedEnv = requiredEnvSchema.parse(process.env);
        console.log('✅ All required environment variables are valid!\n');
        console.log('📋 Configuration Summary:');
        console.log('========================');
        console.log(`Environment: ${validatedEnv.NODE_ENV}`);
        console.log(`Port: ${validatedEnv.PORT}`);
        console.log(`API Version: ${validatedEnv.API_VERSION}`);
        console.log(`Database: ${validatedEnv.MONGODB_URI ? '✅ Configured' : '❌ Not configured'}`);
        console.log(`Redis: ${validatedEnv.REDIS_URL ? '✅ Configured' : '❌ Not configured'}`);
        console.log(`JWT: ${validatedEnv.JWT_SECRET ? '✅ Configured' : '❌ Not configured'}`);
        console.log(`Paystack: ${validatedEnv.PAYSTACK_SECRET_KEY ? '✅ Configured' : '❌ Not configured'}`);
        console.log(`Flutterwave: ${validatedEnv.FLUTTERWAVE_SECRET_KEY ? '✅ Configured' : '❌ Not configured'}`);
        console.log(`NIMC: ${validatedEnv.NIMC_API_KEY ? '✅ Configured' : '❌ Not configured'}`);
        console.log(`BVN: ${validatedEnv.BVN_API_KEY ? '✅ Configured' : '❌ Not configured'}`);
        console.log(`SMTP: ${validatedEnv.SMTP_HOST ? '✅ Configured' : '❌ Not configured'}`);
        console.log(`SMS: ${validatedEnv.SMS_API_KEY ? '✅ Configured' : '❌ Not configured'}`);
        console.log(`Log Level: ${validatedEnv.LOG_LEVEL}`);
        console.log(`Pretty Print: ${validatedEnv.LOG_PRETTY_PRINT}`);
        console.log(`CORS Origins: ${validatedEnv.CORS_ORIGIN}`);
        console.log(`Max File Size: ${validatedEnv.MAX_FILE_SIZE} bytes`);
        console.log(`Upload Path: ${validatedEnv.UPLOAD_PATH}`);
        console.log(`NDPR Compliance: ${validatedEnv.NDPR_COMPLIANCE}`);
        console.log(`CBN Compliance: ${validatedEnv.CBN_COMPLIANCE}`);
        console.log(`Data Retention: ${validatedEnv.DATA_RETENTION_DAYS} days`);
        console.log('\n🎉 Configuration validation completed successfully!');
        return true;
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            console.error('❌ Configuration validation failed!\n');
            console.error('Missing or invalid environment variables:');
            console.error('==========================================');
            error.errors.forEach((err, index) => {
                const path = err.path.join('.');
                console.error(`${index + 1}. ${path}: ${err.message}`);
            });
            console.error('\n💡 To fix these issues:');
            console.error('1. Copy env.example to .env');
            console.error('2. Fill in the required values');
            console.error('3. Ensure all required variables are set');
            console.error('4. Run this validation script again');
        }
        else {
            console.error('❌ Unexpected error during validation:', error);
        }
        return false;
    }
};
exports.validateEnvironment = validateEnvironment;
if (require.main === module) {
    const isValid = validateEnvironment();
    process.exit(isValid ? 0 : 1);
}
