"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSecurityMiddleware = exports.getCSPPolicy = exports.securityHeaders = exports.securityConfig = exports.helmetOptions = exports.corsOptions = void 0;
const helmet_1 = __importDefault(require("helmet"));
const cors_1 = __importDefault(require("cors"));
const env_1 = require("./env");
exports.corsOptions = {
    origin: (0, env_1.getCorsOrigins)(),
    credentials: env_1.env.CORS_CREDENTIALS,
    methods: ['GET', 'POST', 'PUT', 'DELETE', 'PATCH', 'OPTIONS'],
    allowedHeaders: [
        'Origin',
        'X-Requested-With',
        'Content-Type',
        'Accept',
        'Authorization',
        'X-API-Key',
        'X-Request-ID',
    ],
    exposedHeaders: ['X-Total-Count', 'X-Page-Count', 'X-Request-ID'],
    maxAge: 86400,
};
exports.helmetOptions = {
    contentSecurityPolicy: {
        directives: {
            defaultSrc: ["'self'"],
            styleSrc: ["'self'", "'unsafe-inline'"],
            scriptSrc: ["'self'"],
            imgSrc: ["'self'", "data:", "https:"],
            connectSrc: ["'self'"],
            fontSrc: ["'self'"],
            objectSrc: ["'none'"],
            mediaSrc: ["'self'"],
            frameSrc: ["'none'"],
        },
    },
    hsts: {
        maxAge: 31536000,
        includeSubDomains: true,
        preload: true,
    },
    noSniff: true,
    referrerPolicy: { policy: 'strict-origin-when-cross-origin' },
    frameguard: { action: 'deny' },
    xssFilter: true,
    hidePoweredBy: true,
};
exports.securityConfig = {
    trustProxy: true,
    maxRequestSize: env_1.env.MAX_FILE_SIZE,
    uploadSecurity: {
        allowedMimeTypes: [
            'image/jpeg',
            'image/png',
            'image/gif',
            'image/webp',
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        ],
        maxFileSize: env_1.env.MAX_FILE_SIZE,
        uploadPath: env_1.env.UPLOAD_PATH,
    },
    sessionSecurity: {
        secure: env_1.env.NODE_ENV === 'production',
        httpOnly: true,
        sameSite: 'strict',
        maxAge: 24 * 60 * 60 * 1000,
    },
    passwordSecurity: {
        minLength: 8,
        requireUppercase: true,
        requireLowercase: true,
        requireNumbers: true,
        requireSpecialChars: true,
        bcryptRounds: env_1.env.BCRYPT_ROUNDS,
    },
    jwtSecurity: {
        secret: env_1.env.JWT_SECRET,
        expiresIn: env_1.env.JWT_EXPIRES_IN,
        refreshExpiresIn: env_1.env.JWT_REFRESH_EXPIRES_IN,
        issuer: env_1.env.APP_NAME,
        audience: env_1.env.APP_URL,
    },
    apiSecurity: {
        requireApiKey: true,
        apiKeyHeader: 'X-API-Key',
        rateLimitEnabled: true,
        requestLogging: true,
        errorLogging: true,
    },
    compliance: {
        ndpr: env_1.env.NDPR_COMPLIANCE,
        cbn: env_1.env.CBN_COMPLIANCE,
        dataRetentionDays: env_1.env.DATA_RETENTION_DAYS,
        auditLogging: true,
        dataEncryption: true,
    },
};
exports.securityHeaders = {
    'X-Content-Type-Options': 'nosniff',
    'X-Frame-Options': 'DENY',
    'X-XSS-Protection': '1; mode=block',
    'Referrer-Policy': 'strict-origin-when-cross-origin',
    'Permissions-Policy': 'geolocation=(), microphone=(), camera=()',
    'Strict-Transport-Security': 'max-age=31536000; includeSubDomains; preload',
};
const getCSPPolicy = (environment) => {
    const basePolicy = {
        'default-src': ["'self'"],
        'script-src': ["'self'"],
        'style-src': ["'self'", "'unsafe-inline'"],
        'img-src': ["'self'", "data:", "https:"],
        'connect-src': ["'self'"],
        'font-src': ["'self'"],
        'object-src': ["'none'"],
        'media-src': ["'self'"],
        'frame-src': ["'none'"],
    };
    if (environment === 'development') {
        basePolicy['script-src'] = ["'self'", "'unsafe-eval'"];
        basePolicy['style-src'] = ["'self'", "'unsafe-inline'"];
    }
    return basePolicy;
};
exports.getCSPPolicy = getCSPPolicy;
const createSecurityMiddleware = () => {
    return {
        helmet: (0, helmet_1.default)(exports.helmetOptions),
        cors: (0, cors_1.default)(exports.corsOptions),
        trustProxy: (req, _res, next) => {
            req.trustProxy = exports.securityConfig.trustProxy;
            next();
        },
    };
};
exports.createSecurityMiddleware = createSecurityMiddleware;
