"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.gracefulRedisShutdown = exports.redisHelpers = exports.redisManager = exports.RedisManager = exports.redisConfigs = exports.bullConnectionOptions = exports.redisConfig = exports.connectionOptions = void 0;
const redis_1 = require("redis");
const env_1 = require("./env");
const logger_1 = require("./logger");
class InMemoryStore {
    store = new Map();
    timers = new Map();
    static MAX_TIMEOUT_MS = 2_147_483_647;
    scheduleExpiryTimer(key, msRemaining) {
        const chunk = Math.min(msRemaining, InMemoryStore.MAX_TIMEOUT_MS);
        const timer = setTimeout(() => {
            const still = this.store.get(key);
            if (!still) {
                this.timers.delete(key);
                return;
            }
            const left = still.expiry - Date.now();
            if (left <= 0) {
                this.store.delete(key);
                this.timers.delete(key);
            }
            else {
                this.scheduleExpiryTimer(key, left);
            }
        }, chunk);
        this.timers.set(key, timer);
    }
    async get(key) {
        const item = this.store.get(key);
        if (!item)
            return null;
        if (Date.now() > item.expiry) {
            this.store.delete(key);
            return null;
        }
        return item.value;
    }
    async set(key, value) {
        this.store.set(key, { value, expiry: Infinity });
    }
    async setEx(key, ttl, value) {
        const expiry = Date.now() + (ttl * 1000);
        this.store.set(key, { value, expiry });
        const existingTimer = this.timers.get(key);
        if (existingTimer) {
            clearTimeout(existingTimer);
        }
        const msRemaining = ttl * 1000;
        this.scheduleExpiryTimer(key, msRemaining);
    }
    async del(key) {
        const existed = this.store.has(key);
        this.store.delete(key);
        const timer = this.timers.get(key);
        if (timer) {
            clearTimeout(timer);
            this.timers.delete(key);
        }
        return existed ? 1 : 0;
    }
    async exists(key) {
        const item = this.store.get(key);
        if (!item)
            return 0;
        if (Date.now() > item.expiry) {
            this.store.delete(key);
            return 0;
        }
        return 1;
    }
    async expire(key, ttl) {
        const item = this.store.get(key);
        if (!item)
            return 0;
        const expiry = Date.now() + (ttl * 1000);
        this.store.set(key, { value: item.value, expiry });
        const existingTimer = this.timers.get(key);
        if (existingTimer) {
            clearTimeout(existingTimer);
        }
        const msRemaining = ttl * 1000;
        this.scheduleExpiryTimer(key, msRemaining);
        return 1;
    }
    async ttl(key) {
        const item = this.store.get(key);
        if (!item)
            return -2;
        if (Date.now() > item.expiry) {
            this.store.delete(key);
            return -2;
        }
        return Math.floor((item.expiry - Date.now()) / 1000);
    }
    async incr(key) {
        const current = await this.get(key);
        const newValue = current ? parseInt(current) + 1 : 1;
        await this.set(key, newValue.toString());
        return newValue;
    }
    async decr(key) {
        const current = await this.get(key);
        const newValue = current ? parseInt(current) - 1 : -1;
        await this.set(key, newValue.toString());
        return newValue;
    }
    async keys(pattern) {
        const regex = new RegExp(pattern.replace(/\*/g, '.*'));
        return Array.from(this.store.keys()).filter(key => regex.test(key));
    }
    async ping() {
        return 'PONG';
    }
    async info() {
        return `# In-Memory Store\nused_memory:${this.store.size}\nconnected_clients:1`;
    }
    async save() {
    }
    async quit() {
        for (const timer of this.timers.values()) {
            clearTimeout(timer);
        }
        this.timers.clear();
        this.store.clear();
    }
}
exports.connectionOptions = (() => {
    try {
        if (!env_1.env.REDIS_URL)
            return null;
        const u = new URL(env_1.env.REDIS_URL);
        const forceTls = env_1.env.REDIS_TLS === true;
        const urlSuggestsTls = u.protocol === 'rediss:';
        const isTls = forceTls || (urlSuggestsTls && env_1.env.REDIS_FORCE_SSL !== 'false');
        const username = decodeURIComponent(u.username || '');
        const password = decodeURIComponent(u.password || '');
        const host = u.hostname;
        const port = Number(u.port || 6379);
        const actualUrl = (!isTls && urlSuggestsTls)
            ? `redis://${username ? `${username}:${password}@` : ''}${host}:${port}`
            : env_1.env.REDIS_URL;
        const base = {
            url: actualUrl,
            socket: {
                host,
                port,
                ...(isTls ? {
                    tls: {
                        rejectUnauthorized: false,
                        servername: host,
                        secureProtocol: 'TLSv1_2_method'
                    }
                } : {}),
                connectTimeout: 10000,
                keepAlive: 5000,
                reconnectStrategy: (retries) => {
                    if (retries > 10)
                        return new Error('Redis max reconnection attempts reached');
                    return Math.min(retries * 100, 3000);
                },
            },
        };
        if (username)
            base.username = username;
        if (password)
            base.password = password;
        return base;
    }
    catch {
        return null;
    }
})();
(() => {
    if (exports.connectionOptions?.socket) {
        const host = exports.connectionOptions.socket.host;
        const port = exports.connectionOptions.socket.port;
    }
    else if (env_1.env.REDIS_URL) {
        logger_1.logger.warn('REDIS_URL provided but invalid. Will evaluate fallback rules.');
    }
    else {
        logger_1.logger.info('No REDIS_URL provided. Will evaluate fallback rules.');
    }
})();
exports.redisConfig = {
    ...(exports.connectionOptions || { url: 'redis://localhost:6379' }),
    socket: {
        ...(exports.connectionOptions?.socket || {}),
        connectTimeout: 10000,
        keepAlive: 5000,
        reconnectStrategy: (retries) => {
            if (retries > 10) {
                console.error("Redis max reconnection attempts reached");
                return new Error("Redis max reconnection attempts reached");
            }
            return Math.min(retries * 100, 3000);
        },
    },
    ...(env_1.env.REDIS_TLS === true && !exports.connectionOptions?.socket?.tls
        ? {
            tls: {
                rejectUnauthorized: false,
                secureProtocol: 'TLSv1_2_method',
                ciphers: 'ECDHE+AESGCM:ECDHE+CHACHA20:DHE+AESGCM:DHE+CHACHA20:!aNULL:!MD5:!DSS'
            }
        }
        : {}),
};
exports.bullConnectionOptions = (() => {
    if (!exports.connectionOptions?.socket)
        return undefined;
    const sock = exports.connectionOptions.socket;
    const conn = {
        host: sock.host,
        port: sock.port,
        maxRetriesPerRequest: null,
        retryDelayOnFailover: 100,
        enableReadyCheck: false,
        lazyConnect: true,
        connectTimeout: 10000,
        commandTimeout: 5000,
        retryDelayOnClusterDown: 300,
        family: 4,
    };
    if (exports.connectionOptions.username)
        conn.username = exports.connectionOptions.username;
    if (exports.connectionOptions.password)
        conn.password = exports.connectionOptions.password;
    if (sock.tls)
        conn.tls = {
            ...sock.tls,
            servername: sock.host,
            secureProtocol: 'TLSv1_2_method',
            ciphers: 'ECDHE+AESGCM:ECDHE+CHACHA20:DHE+AESGCM:DHE+CHACHA20:!aNULL:!MD5:!DSS'
        };
    return conn;
})();
exports.redisConfigs = {
    main: {
        ...exports.redisConfig,
        database: 0,
    },
    jobs: {
        ...exports.redisConfig,
        database: 1,
    },
    cache: {
        ...exports.redisConfig,
        database: 2,
    },
    sessions: {
        ...exports.redisConfig,
        database: 3,
    },
    rateLimit: {
        ...exports.redisConfig,
        database: 4,
    },
};
class RedisManager {
    static instance;
    clients = new Map();
    fallbackClients = new Map();
    isRedisAvailable = true;
    connectionAttempted = false;
    sharedConnection = null;
    connectionCount = 0;
    maxConnections = 3;
    constructor() { }
    static getInstance() {
        if (!RedisManager.instance) {
            RedisManager.instance = new RedisManager();
        }
        return RedisManager.instance;
    }
    shouldUseFallback() {
        if (env_1.env.REDIS_OPTIONAL) {
            return true;
        }
        if ((env_1.env.NODE_ENV === 'development' || env_1.env.NODE_ENV === 'test') && !env_1.env.REDIS_URL) {
            return true;
        }
        return false;
    }
    async getClient(type = 'main') {
        if (this.shouldUseFallback()) {
            if (!env_1.env.REDIS_URL) {
                if (!this.fallbackClients.has(type)) {
                    const fallbackClient = new InMemoryStore();
                    this.fallbackClients.set(type, fallbackClient);
                    console.log(`Using in-memory fallback for Redis ${type} client (no REDIS_URL provided)`);
                }
                return this.fallbackClients.get(type);
            }
            if (this.connectionCount >= this.maxConnections) {
                console.warn(`Redis connection limit reached (${this.maxConnections}). Using fallback for ${type}.`);
                if (!this.fallbackClients.has(type)) {
                    const fallbackClient = new InMemoryStore();
                    this.fallbackClients.set(type, fallbackClient);
                }
                return this.fallbackClients.get(type);
            }
            if (!this.connectionAttempted) {
                this.connectionAttempted = true;
                try {
                    const client = (0, redis_1.createClient)(exports.redisConfigs.main);
                    client.on("error", (err) => {
                        if (this.shouldUseFallback()) {
                            console.warn(`Redis client error (using fallback):`, err.message || err);
                        }
                        else {
                            console.error(`Redis client error:`, err);
                        }
                        this.isRedisAvailable = false;
                    });
                    client.on("connect", () => {
                        console.log(`Redis client connected (${this.connectionCount + 1}/${this.maxConnections})`);
                        this.isRedisAvailable = true;
                    });
                    client.on("ready", () => {
                        console.log(`Redis client ready`);
                        this.isRedisAvailable = true;
                    });
                    client.on("end", () => {
                        this.connectionCount = Math.max(0, this.connectionCount - 1);
                        if (this.shouldUseFallback()) {
                            console.log(`Redis client disconnected (using fallback)`);
                        }
                        else {
                            console.log(`Redis client disconnected`);
                        }
                        this.isRedisAvailable = false;
                    });
                    client.on("reconnecting", () => {
                        if (this.shouldUseFallback()) {
                            console.log(`Redis client reconnecting (will fallback if fails)`);
                        }
                        else {
                            console.log(`Redis client reconnecting`);
                        }
                    });
                    await Promise.race([
                        client.connect(),
                        new Promise((_, reject) => setTimeout(() => reject(new Error("Connection timeout")), 5000)),
                    ]);
                    this.sharedConnection = client;
                    this.clients.set('main', client);
                    this.connectionCount = 1;
                    this.isRedisAvailable = true;
                    return client;
                }
                catch (error) {
                    const errorMessage = error instanceof Error ? error.message : "Unknown error";
                    console.warn(`Redis connection failed, using in-memory fallback:`, errorMessage);
                    this.isRedisAvailable = false;
                }
            }
            if (!this.isRedisAvailable) {
                if (!this.fallbackClients.has(type)) {
                    const fallbackClient = new InMemoryStore();
                    this.fallbackClients.set(type, fallbackClient);
                    console.log(`Using in-memory fallback for Redis ${type} client`);
                }
                return this.fallbackClients.get(type);
            }
        }
        if (this.sharedConnection && this.isRedisAvailable) {
            return this.sharedConnection;
        }
        if (!this.fallbackClients.has(type)) {
            const fallbackClient = new InMemoryStore();
            this.fallbackClients.set(type, fallbackClient);
            console.log(`Using in-memory fallback for Redis ${type} client`);
        }
        return this.fallbackClients.get(type);
    }
    async closeAll() {
        const redisClosePromises = Array.from(this.clients.values()).map(client => client.quit().catch((err) => console.error('Error closing Redis client:', err)));
        const fallbackClosePromises = Array.from(this.fallbackClients.values()).map(client => client.quit().catch((err) => console.error('Error closing fallback client:', err)));
        await Promise.all([...redisClosePromises, ...fallbackClosePromises]);
        this.clients.clear();
        this.fallbackClients.clear();
    }
    async healthCheck() {
        try {
            const client = await this.getClient('main');
            await client.ping();
            return true;
        }
        catch (error) {
            console.error('Redis health check failed:', error);
            return false;
        }
    }
    async getConnectionInfo(type = 'main') {
        try {
            const client = await this.getClient(type);
            const info = await client.info();
            return info;
        }
        catch (error) {
            console.error(`Failed to get Redis info for ${type}:`, error);
            return null;
        }
    }
    isUsingFallback() {
        return !this.isRedisAvailable && this.shouldUseFallback();
    }
    async get(key) {
        try {
            const client = await this.getClient('cache');
            return await client.get(key);
        }
        catch (error) {
            console.error('Redis get error:', error);
            return null;
        }
    }
    async setex(key, ttl, value) {
        try {
            const client = await this.getClient('cache');
            await client.setEx(key, ttl, value);
        }
        catch (error) {
            console.error('Redis setex error:', error);
        }
    }
    async keys(pattern) {
        try {
            const client = await this.getClient('cache');
            return await client.keys(pattern);
        }
        catch (error) {
            console.error('Redis keys error:', error);
            return [];
        }
    }
}
exports.RedisManager = RedisManager;
exports.redisManager = RedisManager.getInstance();
exports.redisHelpers = {
    async setWithExpiry(key, value, ttl) {
        try {
            const client = await exports.redisManager.getClient('cache');
            await client.setEx(key, ttl, value);
        }
        catch (error) {
            console.error('Redis setWithExpiry error:', error);
        }
    },
    async get(key) {
        try {
            const client = await exports.redisManager.getClient('cache');
            return await client.get(key);
        }
        catch (error) {
            console.error('Redis get error:', error);
            return null;
        }
    },
    async del(key) {
        try {
            const client = await exports.redisManager.getClient('cache');
            return await client.del(key);
        }
        catch (error) {
            console.error('Redis del error:', error);
            return 0;
        }
    },
    async exists(key) {
        try {
            const client = await exports.redisManager.getClient('cache');
            return await client.exists(key);
        }
        catch (error) {
            console.error('Redis exists error:', error);
            return 0;
        }
    },
    async expire(key, ttl) {
        try {
            const client = await exports.redisManager.getClient('cache');
            return await client.expire(key, ttl);
        }
        catch (error) {
            console.error('Redis expire error:', error);
            return 0;
        }
    },
    async ttl(key) {
        try {
            const client = await exports.redisManager.getClient('cache');
            return await client.ttl(key);
        }
        catch (error) {
            console.error('Redis ttl error:', error);
            return -2;
        }
    },
    async incr(key) {
        try {
            const client = await exports.redisManager.getClient('cache');
            return await client.incr(key);
        }
        catch (error) {
            console.error('Redis incr error:', error);
            return 0;
        }
    },
    async decr(key) {
        try {
            const client = await exports.redisManager.getClient('cache');
            return await client.decr(key);
        }
        catch (error) {
            console.error('Redis decr error:', error);
            return 0;
        }
    },
    async set(key, value) {
        try {
            const client = await exports.redisManager.getClient('cache');
            await client.set(key, value);
        }
        catch (error) {
            console.error('Redis set error:', error);
        }
    },
    async keys(pattern) {
        try {
            const client = await exports.redisManager.getClient('cache');
            return await client.keys(pattern);
        }
        catch (error) {
            console.error('Redis keys error:', error);
            return [];
        }
    },
};
const gracefulRedisShutdown = async (signal) => {
    console.log(`Received ${signal}. Closing Redis connections...`);
    try {
        await exports.redisManager.closeAll();
        console.log('Redis connections closed successfully');
        process.exit(0);
    }
    catch (error) {
        console.error('Error during Redis graceful shutdown:', error);
        process.exit(1);
    }
};
exports.gracefulRedisShutdown = gracefulRedisShutdown;
