"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createCustomRateLimit = exports.adminRateLimit = exports.uploadRateLimit = exports.paymentRateLimit = exports.verificationRateLimit = exports.authRateLimit = exports.defaultRateLimit = void 0;
const express_rate_limit_1 = __importDefault(require("express-rate-limit"));
const env_1 = require("./env");
exports.defaultRateLimit = (0, express_rate_limit_1.default)({
    windowMs: env_1.env.RATE_LIMIT_WINDOW_MS,
    max: env_1.env.RATE_LIMIT_MAX_REQUESTS,
    message: {
        error: 'Too many requests from this IP, please try again later.',
        retryAfter: Math.ceil(env_1.env.RATE_LIMIT_WINDOW_MS / 1000),
    },
    standardHeaders: true,
    legacyHeaders: false,
    keyGenerator: (req) => {
        const forwardedFor = req.headers['x-forwarded-for'];
        if (forwardedFor) {
            return forwardedFor.split(',')[0].trim();
        }
        return req.ip || req.connection.remoteAddress || 'unknown';
    },
    handler: (_req, res) => {
        res.status(429).json({
            success: false,
            message: 'Too many requests from this IP, please try again later.',
            retryAfter: Math.ceil(env_1.env.RATE_LIMIT_WINDOW_MS / 1000),
        });
    },
});
exports.authRateLimit = (0, express_rate_limit_1.default)({
    windowMs: 15 * 60 * 1000,
    max: 5,
    message: {
        error: 'Too many authentication attempts, please try again later.',
        retryAfter: 900,
    },
    standardHeaders: true,
    legacyHeaders: false,
    keyGenerator: (req) => {
        const forwardedFor = req.headers['x-forwarded-for'];
        if (forwardedFor) {
            return forwardedFor.split(',')[0].trim();
        }
        return req.ip || req.connection.remoteAddress || 'unknown';
    },
    handler: (_req, res) => {
        res.status(429).json({
            success: false,
            message: 'Too many authentication attempts, please try again later.',
            retryAfter: 900,
        });
    },
});
exports.verificationRateLimit = (0, express_rate_limit_1.default)({
    windowMs: 10 * 60 * 1000,
    max: 5,
    message: {
        error: 'Too many verification attempts, please try again later.',
        retryAfter: 600,
    },
    standardHeaders: true,
    legacyHeaders: false,
    keyGenerator: (req) => {
        const userId = req.user?.userId;
        return userId || (req.ip || req.headers['x-forwarded-for'] || 'unknown');
    },
    handler: (_req, res) => {
        res.setHeader('Retry-After', Math.ceil(600).toString());
        res.status(429).json({
            success: false,
            message: 'Too many verification attempts, please try again later.',
            retryAfter: 600,
        });
    },
});
exports.paymentRateLimit = (0, express_rate_limit_1.default)({
    windowMs: 5 * 60 * 1000,
    max: 10,
    message: {
        error: 'Too many payment requests, please try again later.',
        retryAfter: 300,
    },
    standardHeaders: true,
    legacyHeaders: false,
    handler: (_req, res) => {
        res.status(429).json({
            success: false,
            message: 'Too many payment requests, please try again later.',
            retryAfter: 300,
        });
    },
});
exports.uploadRateLimit = (0, express_rate_limit_1.default)({
    windowMs: 60 * 60 * 1000,
    max: 20,
    message: {
        error: 'Too many file uploads, please try again later.',
        retryAfter: 3600,
    },
    standardHeaders: true,
    legacyHeaders: false,
    handler: (_req, res) => {
        res.status(429).json({
            success: false,
            message: 'Too many file uploads, please try again later.',
            retryAfter: 3600,
        });
    },
});
exports.adminRateLimit = (0, express_rate_limit_1.default)({
    windowMs: 60 * 1000,
    max: 30,
    message: {
        error: 'Too many admin requests, please try again later.',
        retryAfter: 60,
    },
    standardHeaders: true,
    legacyHeaders: false,
    handler: (_req, res) => {
        res.status(429).json({
            success: false,
            message: 'Too many admin requests, please try again later.',
            retryAfter: 60,
        });
    },
});
const createCustomRateLimit = (windowMs, max, message) => {
    return (0, express_rate_limit_1.default)({
        windowMs,
        max,
        message: {
            error: message,
            retryAfter: Math.ceil(windowMs / 1000),
        },
        standardHeaders: true,
        legacyHeaders: false,
        handler: (_req, res) => {
            res.status(429).json({
                success: false,
                message,
                retryAfter: Math.ceil(windowMs / 1000),
            });
        },
    });
};
exports.createCustomRateLimit = createCustomRateLimit;
