"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.queueManager = exports.QueueManager = exports.QUEUE_CONFIGS = exports.DEFAULT_QUEUE_CONFIG = exports.JOB_TYPES = exports.QUEUE_NAMES = void 0;
const bullmq_1 = require("bullmq");
const redis_1 = require("./redis");
const queue_fallback_1 = require("./queue-fallback");
const logger_1 = require("./logger");
const env_1 = require("./env");
exports.QUEUE_NAMES = {
    RECONCILIATION: 'reconciliation',
    CLEANUP: 'cleanup',
    REPORTS: 'reports',
    DISPUTE_ESCALATION: 'dispute-escalation',
    NOTIFICATIONS: 'notifications',
    DATA_RETENTION: 'data-retention'
};
exports.JOB_TYPES = {
    DAILY_RECONCILIATION: 'daily-reconciliation',
    PROVIDER_RECONCILIATION: 'provider-reconciliation',
    OLD_TRANSACTIONS_CLEANUP: 'old-transactions-cleanup',
    OLD_LOGS_CLEANUP: 'old-logs-cleanup',
    OLD_SESSIONS_CLEANUP: 'old-sessions-cleanup',
    DAILY_REPORT: 'daily-report',
    WEEKLY_REPORT: 'weekly-report',
    MONTHLY_REPORT: 'monthly-report',
    DISPUTE_REPORT: 'dispute-report',
    ESCALATE_URGENT_DISPUTES: 'escalate-urgent-disputes',
    DISPUTE_REMINDERS: 'dispute-reminders',
    SEND_EMAIL: 'send-email',
    SEND_SMS: 'send-sms',
    SEND_PUSH: 'send-push',
    RUN_DATA_RETENTION: 'run-data-retention'
};
exports.DEFAULT_QUEUE_CONFIG = {
    name: '',
    concurrency: 1,
    maxAttempts: 3,
    backoffDelay: 5000,
    removeOnComplete: 100,
    removeOnFail: 50,
    delay: 0,
    priority: 0
};
exports.QUEUE_CONFIGS = {
    [exports.QUEUE_NAMES.RECONCILIATION]: {
        ...exports.DEFAULT_QUEUE_CONFIG,
        name: exports.QUEUE_NAMES.RECONCILIATION,
        concurrency: 2,
        maxAttempts: 5,
        backoffDelay: 10000,
        removeOnComplete: 50,
        removeOnFail: 25
    },
    [exports.QUEUE_NAMES.CLEANUP]: {
        ...exports.DEFAULT_QUEUE_CONFIG,
        name: exports.QUEUE_NAMES.CLEANUP,
        concurrency: 1,
        maxAttempts: 3,
        backoffDelay: 30000,
        removeOnComplete: 100,
        removeOnFail: 50
    },
    [exports.QUEUE_NAMES.REPORTS]: {
        ...exports.DEFAULT_QUEUE_CONFIG,
        name: exports.QUEUE_NAMES.REPORTS,
        concurrency: 3,
        maxAttempts: 2,
        backoffDelay: 15000,
        removeOnComplete: 200,
        removeOnFail: 100
    },
    [exports.QUEUE_NAMES.DISPUTE_ESCALATION]: {
        ...exports.DEFAULT_QUEUE_CONFIG,
        name: exports.QUEUE_NAMES.DISPUTE_ESCALATION,
        concurrency: 1,
        maxAttempts: 3,
        backoffDelay: 60000,
        removeOnComplete: 50,
        removeOnFail: 25
    },
    [exports.QUEUE_NAMES.NOTIFICATIONS]: {
        ...exports.DEFAULT_QUEUE_CONFIG,
        name: exports.QUEUE_NAMES.NOTIFICATIONS,
        concurrency: 5,
        maxAttempts: 3,
        backoffDelay: 5000,
        removeOnComplete: 1000,
        removeOnFail: 500
    },
    [exports.QUEUE_NAMES.DATA_RETENTION]: {
        ...exports.DEFAULT_QUEUE_CONFIG,
        name: exports.QUEUE_NAMES.DATA_RETENTION,
        concurrency: 1,
        maxAttempts: 2,
        backoffDelay: 60000,
        removeOnComplete: 50,
        removeOnFail: 25
    }
};
class QueueManager {
    static instance;
    queues = new Map();
    workers = new Map();
    schedulers = new Map();
    useFallback = false;
    constructor() {
        this.useFallback = env_1.env.REDIS_OPTIONAL || ((env_1.env.NODE_ENV === 'development' || env_1.env.NODE_ENV === 'test') && !env_1.env.REDIS_URL);
    }
    static getInstance() {
        if (!QueueManager.instance) {
            QueueManager.instance = new QueueManager();
        }
        return QueueManager.instance;
    }
    async isRedisAvailable() {
        if (!this.useFallback)
            return true;
        try {
            await redis_1.redisManager.getClient('jobs');
            return !redis_1.redisManager.isUsingFallback();
        }
        catch (error) {
            return false;
        }
    }
    shouldUseFallbackImmediately() {
        if (env_1.env.REDIS_OPTIONAL) {
            return true;
        }
        return this.useFallback && (!env_1.env.REDIS_URL || redis_1.redisManager.isUsingFallback());
    }
    async getQueue(name) {
        if (!redis_1.bullConnectionOptions && !this.useFallback) {
            throw new Error('Redis configuration invalid and fallback disabled');
        }
        if (this.shouldUseFallbackImmediately() || !redis_1.bullConnectionOptions) {
            const config = exports.QUEUE_CONFIGS[name] || exports.DEFAULT_QUEUE_CONFIG;
            return queue_fallback_1.fallbackQueueManager.getQueue(name, {
                concurrency: config.concurrency || 1,
                maxAttempts: config.maxAttempts || 3,
                backoffDelay: config.backoffDelay || 1000
            });
        }
        const redisAvailable = await this.isRedisAvailable();
        if (!redisAvailable) {
            const config = exports.QUEUE_CONFIGS[name] || exports.DEFAULT_QUEUE_CONFIG;
            return queue_fallback_1.fallbackQueueManager.getQueue(name, {
                concurrency: config.concurrency || 1,
                maxAttempts: config.maxAttempts || 3,
                backoffDelay: config.backoffDelay || 1000
            });
        }
        if (!this.queues.has(name)) {
            const config = exports.QUEUE_CONFIGS[name] || exports.DEFAULT_QUEUE_CONFIG;
            const queue = new bullmq_1.Queue(name, {
                connection: redis_1.bullConnectionOptions,
                defaultJobOptions: {
                    attempts: config.maxAttempts || 3,
                    backoff: {
                        type: 'exponential',
                        delay: config.backoffDelay || 1000
                    },
                    removeOnComplete: config.removeOnComplete || 10,
                    removeOnFail: config.removeOnFail || 5,
                    delay: config.delay || 0,
                    priority: config.priority || 0
                }
            });
            this.queues.set(name, queue);
            logger_1.logger.info(`Queue created: ${name}`);
        }
        return this.queues.get(name);
    }
    async getWorker(name, processor) {
        if (!redis_1.bullConnectionOptions && !this.useFallback) {
            throw new Error('Redis configuration invalid and fallback disabled');
        }
        if (this.shouldUseFallbackImmediately() || !redis_1.bullConnectionOptions) {
            return queue_fallback_1.fallbackQueueManager.getWorker(name, processor);
        }
        const redisAvailable = await this.isRedisAvailable();
        if (!redisAvailable) {
            return queue_fallback_1.fallbackQueueManager.getWorker(name, processor);
        }
        if (!this.workers.has(name)) {
            const config = exports.QUEUE_CONFIGS[name] || exports.DEFAULT_QUEUE_CONFIG;
            const worker = new bullmq_1.Worker(name, processor, {
                connection: redis_1.bullConnectionOptions,
                concurrency: config.concurrency ?? 1,
                removeOnComplete: (config.removeOnComplete ?? 10),
                removeOnFail: (config.removeOnFail ?? 5)
            });
            worker.on('completed', (job) => {
                logger_1.logger.info(`Job completed: ${job.id} in queue ${name}`);
            });
            worker.on('failed', (job, err) => {
                logger_1.logger.error(`Job failed: ${job?.id} in queue ${name}`, err);
            });
            worker.on('error', (err) => {
                logger_1.logger.error(`Worker error in queue ${name}:`, err);
            });
            this.workers.set(name, worker);
            logger_1.logger.info(`Worker created for queue: ${name}`);
        }
        return this.workers.get(name);
    }
    async getScheduler(name) {
        if (!redis_1.bullConnectionOptions && !this.useFallback) {
            throw new Error('Redis configuration invalid and fallback disabled');
        }
        if (this.shouldUseFallbackImmediately() || !redis_1.bullConnectionOptions) {
            return queue_fallback_1.fallbackQueueManager.getScheduler(name);
        }
        const redisAvailable = await this.isRedisAvailable();
        if (!redisAvailable) {
            return queue_fallback_1.fallbackQueueManager.getScheduler(name);
        }
        if (!this.schedulers.has(name)) {
        }
        return queue_fallback_1.fallbackQueueManager.getScheduler(name);
    }
    async addJob(queueName, jobType, data, options) {
        const queue = await this.getQueue(queueName);
        const job = await queue.add(jobType, data, {
            delay: options?.delay,
            priority: options?.priority,
            attempts: options?.attempts,
            backoff: options?.backoff ? {
                type: 'exponential',
                delay: options.backoff
            } : undefined,
            removeOnComplete: options?.removeOnComplete,
            removeOnFail: options?.removeOnFail
        });
        logger_1.logger.info(`Job added to queue ${queueName}: ${job.id} (${jobType})`);
        return job;
    }
    async addRecurringJob(queueName, jobType, data, pattern, options) {
        const queue = await this.getQueue(queueName);
        if (this.useFallback && !(await this.isRedisAvailable())) {
            logger_1.logger.warn(`Recurring jobs not supported in fallback mode for queue ${queueName}. Adding as regular job.`);
            return await this.addJob(queueName, jobType, data, options);
        }
        const job = await queue.add(jobType, data, {
            repeat: { pattern },
            priority: options?.priority,
            attempts: options?.attempts,
            backoff: options?.backoff ? {
                type: 'exponential',
                delay: options.backoff
            } : undefined
        });
        logger_1.logger.info(`Recurring job added to queue ${queueName}: ${job.id} (${jobType}) with pattern ${pattern}`);
        return job;
    }
    async getQueueStats(queueName) {
        if (this.shouldUseFallbackImmediately() || !redis_1.bullConnectionOptions) {
            return await queue_fallback_1.fallbackQueueManager.getQueueStats(queueName);
        }
        const queue = await this.getQueue(queueName);
        if (this.useFallback && !(await this.isRedisAvailable())) {
            return await queue_fallback_1.fallbackQueueManager.getQueueStats(queueName);
        }
        const [waiting, active, completed, failed, delayed] = await Promise.all([
            queue.getWaiting(),
            queue.getActive(),
            queue.getCompleted(),
            queue.getFailed(),
            queue.getDelayed()
        ]);
        return {
            name: queueName,
            waiting: waiting.length,
            active: active.length,
            completed: completed.length,
            failed: failed.length,
            delayed: delayed.length
        };
    }
    async getAllQueueStats() {
        if (this.shouldUseFallbackImmediately() || !redis_1.bullConnectionOptions) {
            return await queue_fallback_1.fallbackQueueManager.getAllQueueStats();
        }
        if (this.useFallback && !(await this.isRedisAvailable())) {
            return await queue_fallback_1.fallbackQueueManager.getAllQueueStats();
        }
        const stats = [];
        for (const queueName of Object.values(exports.QUEUE_NAMES)) {
            try {
                const stat = await this.getQueueStats(queueName);
                stats.push(stat);
            }
            catch (error) {
                logger_1.logger.error(`Failed to get stats for queue ${queueName}:`, error);
            }
        }
        return stats;
    }
    async cleanupOldJobs(queueName, olderThan = 24 * 60 * 60 * 1000) {
        if (this.shouldUseFallbackImmediately()) {
            logger_1.logger.info(`Cleanup not needed for fallback queue ${queueName}`);
            return 0;
        }
        if (this.useFallback && !(await this.isRedisAvailable())) {
            logger_1.logger.info(`Cleanup not needed for fallback queue ${queueName}`);
            return 0;
        }
        const queue = await this.getQueue(queueName);
        const cutoff = Date.now() - olderThan;
        try {
            const completed = await queue.getCompleted();
            const failed = await queue.getFailed();
            let cleanedCount = 0;
            for (const job of completed) {
                if (job.finishedOn && job.finishedOn < cutoff) {
                    await job.remove();
                    cleanedCount++;
                }
            }
            for (const job of failed) {
                if (job.finishedOn && job.finishedOn < cutoff) {
                    await job.remove();
                    cleanedCount++;
                }
            }
            logger_1.logger.info(`Cleaned up ${cleanedCount} old jobs from queue ${queueName}`);
            return cleanedCount;
        }
        catch (error) {
            logger_1.logger.error(`Failed to cleanup old jobs from queue ${queueName}:`, error);
            throw error;
        }
    }
    async shutdown() {
        logger_1.logger.info('Shutting down queue manager...');
        for (const [name, worker] of this.workers) {
            try {
                await worker.close();
                logger_1.logger.info(`Worker closed for queue: ${name}`);
            }
            catch (error) {
                logger_1.logger.error(`Error closing worker for queue ${name}:`, error);
            }
        }
        for (const [name, scheduler] of this.schedulers) {
            try {
                await scheduler.close();
                logger_1.logger.info(`Scheduler closed for queue: ${name}`);
            }
            catch (error) {
                logger_1.logger.error(`Error closing scheduler for queue ${name}:`, error);
            }
        }
        for (const [name, queue] of this.queues) {
            try {
                await queue.close();
                logger_1.logger.info(`Queue closed: ${name}`);
            }
            catch (error) {
                logger_1.logger.error(`Error closing queue ${name}:`, error);
            }
        }
        try {
            await queue_fallback_1.fallbackQueueManager.shutdown();
        }
        catch (error) {
            logger_1.logger.error('Error shutting down fallback queue manager:', error);
        }
        this.workers.clear();
        this.schedulers.clear();
        this.queues.clear();
        logger_1.logger.info('Queue manager shutdown completed');
    }
}
exports.QueueManager = QueueManager;
exports.queueManager = QueueManager.getInstance();
