"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.fallbackQueueManager = exports.FallbackQueueManager = void 0;
const logger_1 = require("./logger");
class InMemoryQueue {
    name;
    jobs = new Map();
    processors = new Map();
    timers = new Map();
    isProcessing = false;
    concurrency = 1;
    maxAttempts = 3;
    backoffDelay = 5000;
    constructor(name, options = {}) {
        this.name = name;
        this.concurrency = options.concurrency || 1;
        this.maxAttempts = options.maxAttempts || 3;
        this.backoffDelay = options.backoffDelay || 5000;
    }
    async add(jobName, data, options = {}) {
        const jobId = `${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
        const job = {
            id: jobId,
            name: jobName,
            data,
            options,
            createdAt: Date.now(),
            attempts: 0,
            maxAttempts: options.attempts || this.maxAttempts,
            status: options.delay ? 'delayed' : 'waiting'
        };
        this.jobs.set(jobId, job);
        if (options.delay) {
            const timer = setTimeout(() => {
                job.status = 'waiting';
                this.timers.delete(jobId);
                this.processNext();
            }, options.delay);
            this.timers.set(jobId, timer);
        }
        else {
            this.processNext();
        }
        logger_1.logger.info(`Job added to in-memory queue ${this.name}: ${jobId} (${jobName})`);
        return job;
    }
    async process(processor) {
        this.processors.set('default', processor);
        this.processNext();
    }
    async processNext() {
        if (this.isProcessing)
            return;
        const waitingJobs = Array.from(this.jobs.values())
            .filter(job => job.status === 'waiting')
            .sort((a, b) => (b.options.priority || 0) - (a.options.priority || 0));
        if (waitingJobs.length === 0)
            return;
        this.isProcessing = true;
        const jobsToProcess = waitingJobs.slice(0, this.concurrency);
        const promises = jobsToProcess.map(job => this.processJob(job));
        await Promise.all(promises);
        this.isProcessing = false;
        if (Array.from(this.jobs.values()).some(job => job.status === 'waiting')) {
            setImmediate(() => this.processNext());
        }
    }
    async processJob(job) {
        const processor = this.processors.get('default');
        if (!processor) {
            logger_1.logger.error(`No processor found for queue ${this.name}`);
            return;
        }
        job.status = 'active';
        job.attempts++;
        try {
            await processor(job);
            job.status = 'completed';
            job.processedAt = Date.now();
            logger_1.logger.info(`Job completed in queue ${this.name}: ${job.id}`);
        }
        catch (error) {
            logger_1.logger.error(`Job failed in queue ${this.name}: ${job.id}`, error);
            if (job.attempts < job.maxAttempts) {
                job.status = 'delayed';
                const delay = this.backoffDelay * Math.pow(2, job.attempts - 1);
                const timer = setTimeout(() => {
                    job.status = 'waiting';
                    this.timers.delete(job.id);
                    this.processNext();
                }, delay);
                this.timers.set(job.id, timer);
            }
            else {
                job.status = 'failed';
                job.failedAt = Date.now();
            }
        }
    }
    async getWaiting() {
        return Array.from(this.jobs.values()).filter(job => job.status === 'waiting');
    }
    async getActive() {
        return Array.from(this.jobs.values()).filter(job => job.status === 'active');
    }
    async getCompleted() {
        return Array.from(this.jobs.values()).filter(job => job.status === 'completed');
    }
    async getFailed() {
        return Array.from(this.jobs.values()).filter(job => job.status === 'failed');
    }
    async getDelayed() {
        return Array.from(this.jobs.values()).filter(job => job.status === 'delayed');
    }
    async getStats() {
        const jobs = Array.from(this.jobs.values());
        return {
            name: this.name,
            waiting: jobs.filter(job => job.status === 'waiting').length,
            active: jobs.filter(job => job.status === 'active').length,
            completed: jobs.filter(job => job.status === 'completed').length,
            failed: jobs.filter(job => job.status === 'failed').length,
            delayed: jobs.filter(job => job.status === 'delayed').length
        };
    }
    async close() {
        for (const timer of this.timers.values()) {
            clearTimeout(timer);
        }
        this.timers.clear();
        this.jobs.clear();
        this.processors.clear();
        logger_1.logger.info(`In-memory queue ${this.name} closed`);
    }
}
class InMemoryWorker {
    name;
    processor;
    queue;
    constructor(name, processor, queue) {
        this.name = name;
        this.processor = processor;
        this.queue = queue;
    }
    async start() {
        await this.queue.process(this.processor);
        logger_1.logger.info(`In-memory worker started for queue: ${this.name}`);
    }
    async close() {
        await this.queue.close();
        logger_1.logger.info(`In-memory worker closed for queue: ${this.name}`);
    }
}
class InMemoryScheduler {
    name;
    timers = new Map();
    constructor(name) {
        this.name = name;
    }
    async close() {
        for (const timer of this.timers.values()) {
            clearTimeout(timer);
        }
        this.timers.clear();
        logger_1.logger.info(`In-memory scheduler closed for queue: ${this.name}`);
    }
}
class FallbackQueueManager {
    static instance;
    queues = new Map();
    workers = new Map();
    schedulers = new Map();
    constructor() { }
    static getInstance() {
        if (!FallbackQueueManager.instance) {
            FallbackQueueManager.instance = new FallbackQueueManager();
        }
        return FallbackQueueManager.instance;
    }
    getQueue(name, options = {}) {
        if (!this.queues.has(name)) {
            const queue = new InMemoryQueue(name, options);
            this.queues.set(name, queue);
            logger_1.logger.info(`In-memory queue created: ${name}`);
        }
        return this.queues.get(name);
    }
    getWorker(name, processor) {
        if (!this.workers.has(name)) {
            const queue = this.getQueue(name);
            const worker = new InMemoryWorker(name, processor, queue);
            this.workers.set(name, worker);
            worker.start().catch(err => {
                logger_1.logger.error(`Failed to start in-memory worker for queue ${name}:`, err);
            });
            logger_1.logger.info(`In-memory worker created and started for queue: ${name}`);
        }
        return this.workers.get(name);
    }
    getScheduler(name) {
        if (!this.schedulers.has(name)) {
            const scheduler = new InMemoryScheduler(name);
            this.schedulers.set(name, scheduler);
            logger_1.logger.info(`In-memory scheduler created for queue: ${name}`);
        }
        return this.schedulers.get(name);
    }
    async addJob(queueName, jobType, data, options = {}) {
        const queue = this.getQueue(queueName);
        return await queue.add(jobType, data, options);
    }
    async getQueueStats(queueName) {
        const queue = this.getQueue(queueName);
        return await queue.getStats();
    }
    async getAllQueueStats() {
        const stats = [];
        for (const [name, queue] of this.queues) {
            try {
                const stat = await queue.getStats();
                stats.push(stat);
            }
            catch (error) {
                logger_1.logger.error(`Failed to get stats for queue ${name}:`, error);
            }
        }
        return stats;
    }
    async shutdown() {
        logger_1.logger.info('Shutting down fallback queue manager...');
        for (const [name, worker] of this.workers) {
            try {
                await worker.close();
                logger_1.logger.info(`Worker closed for queue: ${name}`);
            }
            catch (error) {
                logger_1.logger.error(`Error closing worker for queue ${name}:`, error);
            }
        }
        for (const [name, scheduler] of this.schedulers) {
            try {
                await scheduler.close();
                logger_1.logger.info(`Scheduler closed for queue: ${name}`);
            }
            catch (error) {
                logger_1.logger.error(`Error closing scheduler for queue ${name}:`, error);
            }
        }
        for (const [name, queue] of this.queues) {
            try {
                await queue.close();
                logger_1.logger.info(`Queue closed: ${name}`);
            }
            catch (error) {
                logger_1.logger.error(`Error closing queue ${name}:`, error);
            }
        }
        this.workers.clear();
        this.schedulers.clear();
        this.queues.clear();
        logger_1.logger.info('Fallback queue manager shutdown completed');
    }
}
exports.FallbackQueueManager = FallbackQueueManager;
exports.fallbackQueueManager = FallbackQueueManager.getInstance();
