"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCorsOrigins = exports.getMongoUri = exports.env = void 0;
const zod_1 = require("zod");
const dotenv_1 = __importDefault(require("dotenv"));
const path_1 = __importDefault(require("path"));
const backendEnvPath = path_1.default.resolve(__dirname, '../..', '.env');
dotenv_1.default.config({ path: backendEnvPath });
const zBooleanFromEnv = zod_1.z.preprocess((val) => val === 'true', zod_1.z.boolean());
const envSchema = zod_1.z.object({
    NODE_ENV: zod_1.z
        .enum(['development', 'production', 'test'])
        .default('development'),
    PORT: zod_1.z.string().pipe(zod_1.z.coerce.number()).default('3000'),
    API_VERSION: zod_1.z.string().default('v1'),
    MONGODB_URI: zod_1.z
        .string()
        .regex(/^mongodb(\+srv)?:\/\/.*/, 'Invalid MongoDB URI'),
    MONGODB_URI_PROD: zod_1.z
        .string()
        .regex(/^mongodb(\+srv)?:\/\/.*/, 'Invalid production MongoDB URI')
        .optional(),
    JWT_SECRET: zod_1.z.string().min(32, 'JWT secret must be at least 32 characters'),
    JWT_EXPIRES_IN: zod_1.z.string().default('7d'),
    JWT_REFRESH_EXPIRES_IN: zod_1.z.string().default('30d'),
    FIELD_ENCRYPTION_KEY: zod_1.z.string().optional(),
    REDIS_URL: zod_1.z
        .union([
        zod_1.z.string().url('Invalid Redis URL'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    REDIS_PASSWORD: zod_1.z.string().optional(),
    REDIS_OPTIONAL: zBooleanFromEnv.default(false),
    REDIS_TLS: zBooleanFromEnv.default(false),
    REDIS_FORCE_SSL: zod_1.z.string().optional(),
    PAYSTACK_SECRET_KEY: zod_1.z
        .union([
        zod_1.z.string().min(1, 'Paystack secret key is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    PAYSTACK_PUBLIC_KEY: zod_1.z
        .union([
        zod_1.z.string().min(1, 'Paystack public key is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    PAYSTACK_WEBHOOK_SECRET: zod_1.z
        .union([
        zod_1.z.string().min(1, 'Paystack webhook secret is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    FLUTTERWAVE_SECRET_KEY: zod_1.z
        .union([
        zod_1.z.string().min(1, 'Flutterwave secret key is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    FLUTTERWAVE_PUBLIC_KEY: zod_1.z
        .union([
        zod_1.z.string().min(1, 'Flutterwave public key is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    ASPFIY_SECRET_KEY: zod_1.z
        .union([
        zod_1.z.string().min(1, 'Aspfiy secret key is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    ASPFIY_PUBLIC_KEY: zod_1.z
        .union([
        zod_1.z.string().min(1, 'Aspfiy public key is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    ASPFIY_WEBHOOK_SECRET: zod_1.z
        .union([
        zod_1.z.string().min(1, 'Aspfiy webhook secret is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    ASPFIY_BASE_URL: zod_1.z
        .union([
        zod_1.z.string().url('Invalid Aspfiy base URL'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    ASPFIY_USE_STUB: zBooleanFromEnv.default(false),
    ASPFIY_FORCE_STUB: zBooleanFromEnv.default(false),
    NIMC_API_KEY: zod_1.z
        .union([
        zod_1.z.string().min(1, 'NIMC API key is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    NIMC_API_URL: zod_1.z
        .union([
        zod_1.z.string().url('Invalid NIMC API URL'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    BVN_API_KEY: zod_1.z
        .union([
        zod_1.z.string().min(1, 'BVN API key is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    BVN_API_URL: zod_1.z
        .union([
        zod_1.z.string().url('Invalid BVN API URL'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    BCRYPT_ROUNDS: zod_1.z.string().pipe(zod_1.z.coerce.number()).default('12'),
    RATE_LIMIT_WINDOW_MS: zod_1.z.string().pipe(zod_1.z.coerce.number()).default('900000'),
    RATE_LIMIT_MAX_REQUESTS: zod_1.z.string().pipe(zod_1.z.coerce.number()).default('100'),
    LOG_LEVEL: zod_1.z.enum(['error', 'warn', 'info', 'debug']).default('info'),
    LOG_PRETTY_PRINT: zBooleanFromEnv.default(true),
    CORS_ORIGIN: zod_1.z
        .string()
        .default('http://localhost:3000,http://localhost:19006'),
    CORS_CREDENTIALS: zBooleanFromEnv.default(true),
    MAX_FILE_SIZE: zod_1.z.string().pipe(zod_1.z.coerce.number()).default('5242880'),
    UPLOAD_PATH: zod_1.z.string().default('./uploads'),
    SMTP_HOST: zod_1.z
        .union([
        zod_1.z.string().min(1, 'SMTP host is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    SMTP_PORT: zod_1.z.string().pipe(zod_1.z.coerce.number()).default('587'),
    SMTP_USER: zod_1.z
        .union([
        zod_1.z.string().email('Invalid SMTP user email'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    SMTP_PASS: zod_1.z
        .union([
        zod_1.z.string().min(1, 'SMTP password is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    SMS_API_KEY: zod_1.z
        .union([
        zod_1.z.string().min(1, 'SMS API key is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    SMS_API_URL: zod_1.z
        .union([
        zod_1.z.string().url('Invalid SMS API URL'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    APP_NAME: zod_1.z.string().default('Unify'),
    APP_URL: zod_1.z.string().url('Invalid app URL'),
    FRONTEND_URL: zod_1.z.string().url('Invalid frontend URL'),
    NDPR_COMPLIANCE: zBooleanFromEnv.default(true),
    CBN_COMPLIANCE: zBooleanFromEnv.default(true),
    DATA_RETENTION_DAYS: zod_1.z.string().pipe(zod_1.z.coerce.number()).default('2555'),
    SENTRY_DSN: zod_1.z.string().optional(),
    NEW_RELIC_LICENSE_KEY: zod_1.z.string().optional(),
    ASB_DATA_BASE_URL: zod_1.z
        .union([
        zod_1.z.string().url('Invalid ASB Data base URL'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    ASB_DATA_API_KEY: zod_1.z
        .union([
        zod_1.z.string().min(1, 'ASB Data API key is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    ASB_DATA_SECRET_KEY: zod_1.z.string().optional(),
    VTU_SANDBOX_MODE: zBooleanFromEnv.default(false),
    VTU_DEFAULT_PROVIDER: zod_1.z.enum(['asbdata', 'mock']).default('asbdata'),
    UNIFY_BASE_URL: zod_1.z.string().url('Invalid Unify base URL').default('https://unifyverifier.com'),
    UNIFY_API_KEY: zod_1.z
        .union([
        zod_1.z.string().min(1, 'Unify API key is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    UNIFY_WEBHOOK_SECRET: zod_1.z
        .union([
        zod_1.z.string().min(1, 'Unify webhook secret is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    S8V_BASE_URL: zod_1.z.string().url('Invalid S8V base URL').default('https://www.s8v.ng/api'),
    S8V_API_KEY: zod_1.z
        .union([
        zod_1.z.string().min(1, 'S8V API key is required'),
        zod_1.z.literal(''),
        zod_1.z.undefined(),
    ])
        .optional(),
    VERIFICATION_SANDBOX_MODE: zBooleanFromEnv.default(false),
});
const parseEnv = () => {
    try {
        return envSchema.parse(process.env);
    }
    catch (error) {
        if (error instanceof zod_1.z.ZodError) {
            const missingVars = error.errors.map((err) => `${err.path.join('.')}: ${err.message}`);
            throw new Error(`Environment validation failed:\n${missingVars.join('\n')}`);
        }
        throw error;
    }
};
exports.env = parseEnv();
const getMongoUri = () => {
    if (exports.env.NODE_ENV === 'production' && exports.env.MONGODB_URI_PROD) {
        return exports.env.MONGODB_URI_PROD;
    }
    return exports.env.MONGODB_URI;
};
exports.getMongoUri = getMongoUri;
const mongoUri = (0, exports.getMongoUri)();
if (!mongoUri || !/^mongodb(\+srv)?:\/\/.*/.test(mongoUri)) {
    console.error('[ENV ERROR] Invalid or missing MongoDB URI. Check your .env file and environment variables.');
    process.exit(1);
}
else {
    console.log('[ENV CHECK] MongoDB URI loaded:', mongoUri.replace(/\/\/.*@/, '//[REDACTED]@'));
}
const getCorsOrigins = () => {
    return exports.env.CORS_ORIGIN.split(',').map((origin) => origin.trim());
};
exports.getCorsOrigins = getCorsOrigins;
