"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.gracefulShutdown = exports.checkDatabaseHealth = exports.getDatabaseStatus = exports.disconnectDatabase = exports.connectDatabase = exports.mongooseConfig = exports.databaseConfig = exports.mongoOptions = void 0;
const mongoose_1 = __importDefault(require("mongoose"));
const env_1 = require("./env");
exports.mongoOptions = {
    maxPoolSize: 10,
    minPoolSize: 2,
    maxIdleTimeMS: 30000,
    w: 'majority',
    wtimeoutMS: 10000,
    readPreference: 'primaryPreferred',
    authSource: 'admin',
    connectTimeoutMS: 10000,
    socketTimeoutMS: 45000,
    bufferCommands: true,
    autoIndex: env_1.env.NODE_ENV === 'development',
};
exports.databaseConfig = {
    uri: (0, env_1.getMongoUri)(),
    options: exports.mongoOptions,
    events: {
        connected: 'MongoDB connected successfully',
        disconnected: 'MongoDB disconnected',
        error: 'MongoDB connection error',
        reconnected: 'MongoDB reconnected',
    },
    retry: {
        attempts: 5,
        delay: 1000,
        backoff: 2,
    },
    healthCheck: {
        enabled: true,
        interval: 30000,
        timeout: 5000,
    },
    backup: {
        enabled: env_1.env.NODE_ENV === 'production',
        schedule: '0 2 * * *',
        retention: 7,
    },
    monitoring: {
        enabled: true,
        slowQueryThreshold: 100,
        logQueries: env_1.env.NODE_ENV === 'development',
    },
};
exports.mongooseConfig = {
    plugins: [],
    schemaOptions: {
        timestamps: true,
        toJSON: {
            transform: (_doc, ret) => {
                ret.id = ret._id;
                delete ret._id;
                delete ret.__v;
                return ret;
            },
        },
        toObject: {
            transform: (_doc, ret) => {
                ret.id = ret._id;
                delete ret._id;
                delete ret.__v;
                return ret;
            },
        },
    },
    modelOptions: {},
};
const connectDatabase = async () => {
    try {
        mongoose_1.default.set('strictQuery', false);
        await mongoose_1.default.connect(exports.databaseConfig.uri, exports.databaseConfig.options);
        await mongoose_1.default.connect((0, env_1.getMongoUri)());
        let attempts = 0;
        const { attempts: maxAttempts, delay, backoff } = exports.databaseConfig.retry;
        while (attempts < maxAttempts) {
            try {
                await mongoose_1.default.connect(exports.databaseConfig.uri, exports.databaseConfig.options);
                console.log(exports.databaseConfig.events.connected);
                return;
            }
            catch (error) {
                attempts++;
                console.error(exports.databaseConfig.events.error, error);
                if (attempts >= maxAttempts)
                    throw error;
                await new Promise(r => setTimeout(r, delay * attempts * backoff));
            }
        }
        console.log(exports.databaseConfig.events.connected);
    }
    catch (error) {
        console.error(exports.databaseConfig.events.error, error);
        throw error;
    }
};
exports.connectDatabase = connectDatabase;
const disconnectDatabase = async () => {
    try {
        await mongoose_1.default.disconnect();
        console.log(exports.databaseConfig.events.disconnected);
    }
    catch (error) {
        console.error('Error disconnecting from MongoDB:', error);
        throw error;
    }
};
exports.disconnectDatabase = disconnectDatabase;
const getDatabaseStatus = () => {
    const state = mongoose_1.default.connection.readyState;
    const states = {
        0: 'disconnected',
        1: 'connected',
        2: 'connecting',
        3: 'disconnecting',
    };
    return states[state] || 'unknown';
};
exports.getDatabaseStatus = getDatabaseStatus;
const checkDatabaseHealth = async () => {
    try {
        const db = mongoose_1.default.connection.db;
        if (!db) {
            return false;
        }
        await db.admin().ping();
        return true;
    }
    catch (error) {
        console.error('Database health check failed:', error);
        return false;
    }
};
exports.checkDatabaseHealth = checkDatabaseHealth;
const gracefulShutdown = async (signal) => {
    console.log(`Received ${signal}. Closing database connection...`);
    try {
        await (0, exports.disconnectDatabase)();
        console.log('Database connection closed successfully');
        process.exit(0);
    }
    catch (error) {
        console.error('Error during graceful shutdown:', error);
        process.exit(1);
    }
};
exports.gracefulShutdown = gracefulShutdown;
